<?php
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

require DIR_SYSTEM . 'library/PHPMailer/Exception.php';
require DIR_SYSTEM . 'library/PHPMailer/PHPMailer.php';
require DIR_SYSTEM . 'library/PHPMailer/SMTP.php';

class ControllerPayrollBankLetter extends HController {

    protected function getAlias() {
        return 'payroll/bank_letter';
    }

    protected function getPrimaryKey() {
        return 'bank_letter_id';
    }

    protected function getList() {
        parent::getList();

        $this->data['action_ajax'] = $this->url->link($this->getAlias() . '/getAjaxLists', 'token=' . $this->session->data['token'], 'SSL');
        $this->data['href_custom_excel'] = $this->url->link($this->getAlias() . '/customExcel', 'token=' . $this->session->data['token'], 'SSL');

        $this->model['bank_letter_custom_format'] = $this->load->model('payroll/bank_letter_custom_format');
        $formats = $this->model['bank_letter_custom_format']->getDistinctRows(['id','report_title'],['company_id' => $this->session->data['company_id']]);
        $this->data['excel_formats'] = $formats;

        $this->response->setOutput($this->render());
    }

    public function getAjaxLists() {
        $this->model['bank_letter_custom_format'] = $this->load->model('payroll/bank_letter_custom_format');
        $formats = $this->model['bank_letter_custom_format']->getRows(['company_id' => $this->session->data['company_id']]);

        $lang = $this->load->language($this->getAlias());
        $this->model[$this->getAlias()] = $this->load->model($this->getAlias());
        $data = array();
        $aColumns = array('action', 'document_date', 'document_identity', 'document_title', 'cheque_date', 'cheque_no', 'total_employee','total_amount','created_at','check_box');

        /*
         * Paging
         */
        $sLimit = "";
        if (isset($_GET['iDisplayStart']) && $_GET['iDisplayLength'] != '-1') {
            $data['criteria']['start'] = $_GET['iDisplayStart'];
            $data['criteria']['limit'] = $_GET['iDisplayLength'];
        }

        /*
         * Ordering
         */
        $sOrder = "";
        if (isset($_GET['iSortCol_0'])) {
            $sOrder = " ORDER BY  ";
            for ($i = 0; $i < intval($_GET['iSortingCols']); $i++) {
                if ($_GET['bSortable_' . intval($_GET['iSortCol_' . $i])] == "true") {
                    $sOrder .= "`" . $aColumns[intval($_GET['iSortCol_' . $i])] . "` " .
                        ($_GET['sSortDir_' . $i] === 'asc' ? 'asc' : 'desc') . ", ";
                }
            }

            $sOrder = substr_replace($sOrder, "", -2);
            if ($sOrder == " ORDER BY") {
                $sOrder = "";
            }
            $data['criteria']['orderby'] = $sOrder;
        }


        /*
         * Filtering
         * NOTE this does not match the built-in DataTables filtering which does it
         * word by word on any field. It's possible to do here, but concerned about efficiency
         * on very large tables, and MySQL's regex functionality is very limited
         */
        $arrWhere = array();
        $arrWhere[] = "`company_id` = '".$this->session->data['company_id']."'";
        $arrWhere[] = "`fiscal_year_id` = '".$this->session->data['fiscal_year_id']."'";
        if (isset($_GET['sSearch']) && $_GET['sSearch'] != "") {
            $arrSSearch = array();
            for ($i = 0; $i < count($aColumns); $i++) {
                if (isset($_GET['bSearchable_' . $i]) && $_GET['bSearchable_' . $i] == "true" && $_GET['sSearch'] != '') {
                    $arrSSearch[] = "LOWER(`" . $aColumns[$i] . "`) LIKE '%" . $this->db->escape(strtolower($_GET['sSearch'])) . "%'";
                }
            }
            if(!empty($arrSSearch)) {
                $arrWhere[] = '(' . implode(' OR ', $arrSSearch) . ')';
            }
        }

        /* Individual column filtering */
        for ($i = 0; $i < count($aColumns); $i++) {
            if (isset($_GET['bSearchable_' . $i]) && $_GET['bSearchable_' . $i] == "true" && $_GET['sSearch_' . $i] != '') {
                $arrWhere[] = "LOWER(`" . $aColumns[$i] . "`) LIKE '%" . $this->db->escape(strtolower($_GET['sSearch_' . $i])) . "%' ";
            }
        }

        if (!empty($arrWhere)) {
            //$data['filter']['RAW'] = substr($sWhere, 5, strlen($sWhere) - 5);
            $data['filter']['RAW'] = implode(' AND ', $arrWhere);
        }

        //d($data, true);
        $results = $this->model[$this->getAlias()]->getLists($data);
        $iFilteredTotal = $results['total'];
        $iTotal = $results['table_total'];


        /*
         * Output
         */
        $output = array(
            "sEcho" => intval($_GET['sEcho']),
            "iTotalRecords" => $iTotal,
            "iTotalDisplayRecords" => $iFilteredTotal,
            "aaData" => array()
        );

        foreach ($results['lists'] as $aRow) {
            $row = array();
            $actions = array();

            $actions[] = array(
                'text' => $lang['edit'],
                'href' => $this->url->link($this->getAlias() . '/update', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $aRow[$this->getPrimaryKey()], 'SSL'),
                'btn_class' => 'btn btn-xs btn-primary',
                'class' => 'fa fa-pencil'
            );
            if($aRow['letter_type']=='RTGS') {
                $actions[] = array(
                    'text' => $lang['print'],
                    'target' => '_blank',
                    'href' => $this->url->link($this->getAlias() . '/printRTGS', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $aRow[$this->getPrimaryKey()], 'SSL'),
                    'btn_class' => 'btn btn-xs btn-info',
                    'class' => 'fa fa-print'
                );
            } elseif($aRow['letter_type']=='IFT') {
                $actions[] = array(
                    'text' => $lang['print'],
                    'target' => '_blank',
                    'href' => $this->url->link($this->getAlias() . '/printIFT', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $aRow[$this->getPrimaryKey()], 'SSL'),
                    'btn_class' => 'btn btn-xs btn-info',
                    'class' => 'fa fa-print'
                );
            } else {
                $actions[] = array(
                    'text' => $lang['print'],
                    'target' => '_blank',
                    'href' => $this->url->link($this->getAlias() . '/printTransfer', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $aRow[$this->getPrimaryKey()], 'SSL'),
                    'btn_class' => 'btn btn-xs btn-info',
                    'class' => 'fa fa-print'
                );
            }

            $actions[] = array(
                'text' => $lang['excel'],
                'target' => '_blank',
                'href' => $this->url->link($this->getAlias() . '/downloadExcel', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $aRow[$this->getPrimaryKey()], 'SSL'),
                'btn_class' => 'btn btn-xs btn-success',
                'class' => 'fa fa-download'
            );
            if($formats) {
                $href_download_custom_excel = $this->url->link($this->getAlias() . '/downloadCustomExcel', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $aRow[$this->getPrimaryKey()], 'SSL');
                $actions[] = array(
                    'text' => $lang['excel'],
                    'href' => 'javascript:void(0);',
                    'click' => "selectFormat('$href_download_custom_excel')",
                    'btn_class' => 'btn btn-xs btn-success',
                    'class' => 'fa fa-file-excel-o'
                );
            }

            $actions[] = array(
                'text' => $lang['sms'],
                'target' => '_blank',
                'href' => $this->url->link($this->getAlias() . '/viewSMS', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $aRow[$this->getPrimaryKey()], 'SSL'),
                'btn_class' => 'btn btn-xs btn-success',
                'class' => 'fa fa-commenting-o'
            );

            $actions[] = array(
                'text' => $lang['email'],
                'target' => '_blank',
                'href' => $this->url->link($this->getAlias() . '/viewEmail', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $aRow[$this->getPrimaryKey()], 'SSL'),
                'btn_class' => 'btn btn-xs btn-success',
                'class' => 'fa fa-at'
            );

            if($aRow['is_post'] != 1) {
                $actions[] = array(
                    'text' => $lang['post'],
                    'target' => '_blank',
                    'href' => $this->url->link($this->getAlias() . '/post', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $aRow[$this->getPrimaryKey()], 'SSL'),
                    'btn_class' => 'btn btn-xs btn-info',
                    'class' => 'fa fa-thumbs-up'
                );

                $actions[] = array(
                    'text' => $lang['delete'],
                    'href' => 'javascript:void(0);',
                    'click' => "ConfirmDelete('" . $this->url->link($this->getAlias() . '/delete', 'token=' . $this->session->data['token'] . '&id=' . $aRow[$this->getPrimaryKey()], 'SSL') . "')",
                    'btn_class' => 'btn btn-xs btn-danger',
                    'class' => 'fa fa-times'
                );
            }


            $strAction = '';
            foreach ($actions as $action) {
                $strAction .= '<a '.(isset($action['target'])?'target="'.$action['target'].'"':'').' '.(isset($action['btn_class'])?'class="'.$action['btn_class'].'"':'').' href="' . $action['href'] . '" data-toggle="tooltip" title="' . $action['text'] . '" ' . (isset($action['click']) ? 'onClick="' . $action['click'] . '"' : '') . '>';
                if (isset($action['class'])) {
                    $strAction .= '<span class="' . $action['class'] . '"></span>';
                } else {
                    $strAction .= $action['text'];
                }
                $strAction .= '</a>&nbsp;';
            }

            for ($i = 0; $i < count($aColumns); $i++) {
                if ($aColumns[$i] == 'action') {
                    $row[] = $strAction;
                } elseif ($aColumns[$i] == 'created_at') {
                    $row[] = stdDateTime($aRow['created_at']);
                } elseif ($aColumns[$i] == 'document_date') {
                    $row[] = stdDate($aRow['document_date']);
                } elseif ($aColumns[$i] == 'cheque_date') {
                    $row[] = stdDate($aRow['cheque_date']);
                } else {
                    $row[] = $aRow[$aColumns[$i]];
                }

            }
            $output['aaData'][] = $row;
        }

        echo json_encode($output);
    }

    protected function getForm() {
        parent::getForm();

        $this->model['bank_letter'] = $this->load->model('payroll/bank_letter');
        $this->model['department'] = $this->load->model('setup/department');
        $this->model['employee_registration'] = $this->load->model('employee/employee_registration');
        $this->model['company_account'] = $this->load->model('setup/company_account');
        $this->model['allowance_type'] = $this->load->model('setup/special_allowance_type');
        $this->model['user_access'] = $this->load->model('user/user_access');

        $company_ids = $this->model['user_access']->getArrays('company_id','company_id',array('user_id'=> $this->session->data['user_id']));
        $allowance_types = $this->model['allowance_type']->getRows(array(), array('special_allowance_type'));

        $source_types = array(
            'payroll' => 'Payroll',
            'sp_enayat' => 'Special Enayat',
        );
        foreach($allowance_types as $allowance_type) {
            $source_types[$allowance_type['special_allowance_type_id']] = $allowance_type['special_allowance_type'];
        }


        $this->data['source_types']  = $source_types;
        $this->data['employee_banks'] = $this->model['bank_letter']->getEmployeeBanks();
        $this->data['departments'] = $this->model['department']->getRows(array('company_id' => $this->session->data['company_id']), array('department_name'));
        $this->data['employees'] = $this->model['employee_registration']->getRows(array('company_id' => $this->session->data['company_id']), array('employee_name'));
        
        $filter = [];
        if($this->session->data['is_all_company_account']!=1)
            $filter['company_id'] = $this->session->data['company_id'];
        $this->data['company_accounts'] = $this->model['company_account']->getRows($filter, array('account_title'));

        $this->data['document_date'] = stdDate();
        $this->data['is_edit'] = 0;
        if (isset($this->request->get['bank_letter_id']) && ($this->request->server['REQUEST_METHOD'] != 'POST')) {
            $this->data['is_edit'] = 1;
            $result = $this->model[$this->getAlias()]->getRow(array('bank_letter_id' => $this->request->get['bank_letter_id']));
            foreach($result as $field => $value) {
                if(($field=='document_date' ||  $field=='source_date' ||  $field=='cheque_date' ) && $value != '') {
                    $this->data[$field] = stdDate($value);
                } else {
                    $this->data[$field] = $value;
                }
            }

            $this->model['bank_letter_detail'] = $this->load->model('payroll/bank_letter_detail');
            $details = $this->model['bank_letter_detail']->getRows(array('bank_letter_id' => $result['bank_letter_id']),['employee_name']);
            //d($details, true);
            foreach($details as $detail) {
                $detail['cheque_date'] = stdDate($detail['cheque_date']);

                $this->data['details'][] = $detail;
            }
        }

        $this->data['strValidation']="{
            'rules':{
		        'document_date': {'required':true},
		        'document_title': {'required':true},
		        'source_type': {'required':true},
		        'source_date': {'required':true},
		        'total_employee': {'required':true,'min':1},
		        'total_amount': {'required':true,'min':1},
            },
        }";

        $this->data['href_get_data'] = $this->url->link($this->getAlias() . '/getData', 'token=' . $this->session->data['token'] . '&bank_letter_id=' . $this->request->get['bank_letter_id']);
        $this->response->setOutput($this->render());
    }

    public function getData() {
        $post = $this->request->post;
        if($post['source_type']=='payroll') {
            $this->model['bank_letter'] = $this->load->model('payroll/bank_letter');
            $post['company_id'] = $this->session->data['company_id'];
            $post['fiscal_year_id'] = $this->session->data['fiscal_year_id'];
            $post['source_date'] = MySqlDate($post['source_date']);

            if($post['show_deduction']=='Yes') {
                $accounts = $this->model['bank_letter']->getPayrollDeductions($post['source_date'], $post['company_id']);
            } else {
                $accounts = [];
            }
            $rows = $this->model['bank_letter']->getPayrollData($post);
            $rows = array_merge($accounts, $rows);
        } elseif($post['source_type']=='sp_enayat') {
            $post['company_id'] = $this->session->data['company_id'];
            $post['fiscal_year_id'] = $this->session->data['fiscal_year_id'];
            $post['source_date'] = MySqlDate($post['source_date']);

            $this->model['bank_letter'] = $this->load->model('payroll/bank_letter');
            $rows = $this->model['bank_letter']->getSpecialEnayatData($post);
        } else {
            $post['company_id'] = $this->session->data['company_id'];
            $post['fiscal_year_id'] = $this->session->data['fiscal_year_id'];
            $post['special_allowance_type_id'] = $post['source_type'];
            $post['source_date'] = MySqlDate($post['source_date']);

            $this->model['bank_letter'] = $this->load->model('payroll/bank_letter');
            $rows = $this->model['bank_letter']->getSpecialAllowanceData($post);
        }

        $json = array(
            'success' => true,
            'employees' => $rows
        );

        echo json_encode($json);
    }

    protected function insertData($data) {
        $session = $this->session->data;
        $document_prefix = 'BL'.$session['fy_code'].'-';

        $document = $this->model[$this->getAlias()]->getNextNumber($document_prefix);
        $data['document_prefix'] = $document['document_prefix'];
        $data['document_no'] = $document['document_no'];
        $data['document_identity'] = $document['document_identity'];
        $data['document_date']=MySqlDate($data['document_date']);
        $data['company_id'] = $this->session->data['company_id'];
        $data['fiscal_year_id'] = $this->session->data['fiscal_year_id'];
        $data['source_date']=MySqlDate($data['source_date']);
        $data['department_id']=implode(',', $data['department_id']);
        $data['cheque_date']=MySqlDate($data['cheque_date']);

        $this->model['payroll'] = $this->load->model('payroll/payroll');

        $bank_letter_id = $this->model[$this->getAlias()]->add($this->getAlias(), $data);
        $this->model['bank_letter_detail'] = $this->load->model('payroll/bank_letter_detail');
        $arrPayrolls = [];
        $total_amount =0;
        foreach($data['employees'] as $detail) {
            $detail['bank_letter_id'] = $bank_letter_id;
            $detail['cheque_date'] = MySqlDate($detail['cheque_date']);

            $this->model['bank_letter_detail']->add($this->getAlias(), $detail);
            $this->model['payroll']->edit($this->getAlias(), $detail['source_table_id'], ['deposit_slip_id' => $bank_letter_id]);
            $arrPayrolls[] = $detail['source_table_id'];
            $total_amount += $detail['amount'];
        }

        //d([$data['source_type'], $arrPayrolls, $accounts], true);

        return $bank_letter_id;
    }

    protected function updateData($primary_key, $data) {
        $data['company_id'] = $this->session->data['company_id'];
        $data['fiscal_year_id'] = $this->session->data['fiscal_year_id'];
        $data['bank_letter_date']=MySqlDate($data['bank_letter_date']);
        $data['document_date']=MySqlDate($data['document_date']);
        $data['source_date']=MySqlDate($data['source_date']);
        $data['cheque_date']=MySqlDate($data['cheque_date']);

        $this->model['payroll'] = $this->load->model('payroll/payroll');

        $bank_letter_id = $this->model[$this->getAlias()]->edit($this->getAlias(), $primary_key, $data);
        $this->model['bank_letter_detail'] = $this->load->model('payroll/bank_letter_detail');
        $this->model['bank_letter_detail']->deleteBulk($this->getAlias(), array('bank_letter_id' => $bank_letter_id));

        foreach($data['employees'] as $detail) {
            $detail['bank_letter_id'] = $bank_letter_id;
            $detail['cheque_date'] = MySqlDate($detail['cheque_date']);

            $this->model['bank_letter_detail']->add($this->getAlias(), $detail);
            $this->model['payroll']->edit($this->getAlias(), $detail['source_table_id'], ['deposit_slip_id' => $bank_letter_id]);
        }

        return $bank_letter_id;
    }

    protected function deleteData($primary_key) {
        $this->model['bank_letter_detail'] = $this->load->model('payroll/bank_letter_detail');
        $this->model['bank_letter_detail']->deleteBulk($this->getAlias(), array('bank_letter_id' => $primary_key));
        $this->model[$this->getAlias()]->delete($this->getAlias(), $primary_key);
    }

    public function printTransfer() {
        $bank_letter_id = $this->request->get['bank_letter_id'];
        $this->model['company_account'] = $this->load->model('setup/company_account');

        $lang = $this->load->language('payroll/bank_letter');
        $this->model['bank_letter'] = $this->load->model('payroll/bank_letter');
        $this->model['bank_letter_detail'] = $this->load->model('payroll/bank_letter_detail');

        $voucher = $this->model['bank_letter']->getRow(array('bank_letter_id' => $bank_letter_id));
        $company_account = $this->model['company_account']->getRow(array('company_account_id'=>$voucher['company_account_id']));
        // d($voucher,true);

        $details = $this->model['bank_letter_detail']->getRows(array('bank_letter_id' => $bank_letter_id), array('employee_name'));

        $pdf = new PDF('P', PDF_UNIT, 'A4', true, 'UTF-8', false);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-Bold.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-BoldItalic.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-ExtraBold.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-ExtraBoldItalic.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-Italic.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-Light.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-LightItalic.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-Regular.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-SemiBold.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-SemiBoldItalic.ttf', 'TrueTypeUnicode', '', 96);
        // set document information
        $pdf->SetAuthor('Huzaifa Khambaty');
        $pdf->SetTitle('Bank Letter:'.$voucher['bank_letter_no'].'@'.date('YmdHis'));
        $pdf->SetSubject('Bank Letter');

        // set margins
        //$pdf->SetMargins(PDF_MARGIN_LEFT, PDF_MARGIN_TOP, PDF_MARGIN_RIGHT);
        $pdf->SetMargins(PDF_MARGIN_LEFT, 10, PDF_MARGIN_RIGHT);
        $pdf->SetHeaderMargin(0);
        $pdf->SetFooterMargin(PDF_MARGIN_FOOTER);

        $pdf->footer = array(
            'bank_letter_title' => $voucher['bank_letter_title']
        );
        $pdf->lang = $lang;
        // set auto page breaks
        $pdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM);

        // add a page
        $pdf->AddPage();
        $pdf->SetFont('freesans','',10);
        $pdf->Cell(30,7,stdDate($voucher['bank_letter_date']));
        $pdf->ln(7);
        $pdf->Cell(60,7,$company_account['bank_name'].',');
        $pdf->ln(7);
        $pdf->Cell(60,7,$company_account['bank_branch']);
        $pdf->ln(7);
        $pdf->Cell(60,7,'Letter Ref: '.$voucher['document_identity']);
        $pdf->ln(14);
        $pdf->Cell(30,7,'Dear Sir,');
        $pdf->ln(7);
        $pdf->Cell(15,7,'');
        $pdf->Cell(150,7,'We are enclosing herewith Cheque No. '.$voucher['cheque_no'].'  Dated: '.stdDate($voucher['cheque_date']));
        $pdf->ln(7);
        $pdf->Cell(15,7,'');
        $pdf->Cell(150,7,'for Rs. '.number_format($voucher['total_amount'],2).' drawn on '.$company_account['bank_name'].' being '.$voucher['document_title'].'.');
        $pdf->ln(7);
        $pdf->Cell(15,7,'');
        $pdf->Cell(150,7,'kindly credit the following accounts');
        $pdf->ln(14);
        $pdf->SetFont('freesans','B',9);
        $pdf->Cell(20,7,'SNo.','TB','','C');
        $pdf->Cell(105,7,'Account Title','TB','','L');
        $pdf->Cell(25,7,'Account No.','TB','','R');
        $pdf->Cell(30,7,'Rupees','TB','','R');
        $pdf->SetFont('freesans','',9);
        $sr_no=0;
        foreach($details as $detail) {
            $sr_no++;
            $pdf->ln(7);
            $pdf->Cell(20,7,$sr_no,'B','','C');
            $pdf->Cell(105,7,$detail['bank_title'],'B','','L');
            $pdf->Cell(25,7,$detail['bank_iban'],'B','','R');
            $pdf->Cell(30,7,number_format($detail['amount'],2),'B','','R');
        }
        $pdf->ln(14);
        $pdf->Cell(125,7,'','','','R');
        $pdf->Cell(25,7,'Total Rupees:','TB','','R');
        $pdf->Cell(30,7,number_format($voucher['total_amount'],2),'TB','','R');
        $pdf->ln(8);
        $pdf->Cell(125,7,'','','','R');
        $pdf->Cell(25,7,'Yours truly','T','','LB');
        $pdf->Cell(30,7,'','T','','R');
        $pdf->ln(20);
        $pdf->Cell(125,7,'','','','R');
        if($this->session->data['parent_company_id'] != 0) {
            $pdf->Cell(55,7,'For DAWAT-E-HADIYAH','','','L');
        } else {
            $pdf->Cell(55,7,$this->session->data['company_name'],'','','R');
        }

        $pdf->Output('Bank Letter:'.$voucher['bank_letter_no'].'@'.date('YmdHis').'.pdf', 'I');
    }

    public function printIFT() {
        $bank_letter_id = $this->request->get['bank_letter_id'];
        $this->model['company_account'] = $this->load->model('setup/company_account');

        $lang = $this->load->language('payroll/bank_letter');
        $this->model['bank_letter'] = $this->load->model('payroll/bank_letter');
        $this->model['bank_letter_detail'] = $this->load->model('payroll/bank_letter_detail');

        $voucher = $this->model['bank_letter']->getRow(array('bank_letter_id' => $bank_letter_id));
        $company_account = $this->model['company_account']->getRow(array('company_account_id'=>$voucher['company_account_id']));
        // d($voucher,true);

        $details = $this->model['bank_letter_detail']->getRows(array('bank_letter_id' => $bank_letter_id), array('employee_name'));

        $pdf = new PDF('P', PDF_UNIT, 'A4', true, 'UTF-8', false);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-Bold.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-BoldItalic.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-ExtraBold.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-ExtraBoldItalic.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-Italic.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-Light.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-LightItalic.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-Regular.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-SemiBold.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-SemiBoldItalic.ttf', 'TrueTypeUnicode', '', 96);
        // set document information
        $pdf->SetAuthor('Huzaifa Khambaty');
        $pdf->SetTitle('Bank Letter:'.$voucher['bank_letter_no'].'@'.date('YmdHis'));
        $pdf->SetSubject('Bank Letter');

        // set margins
        //$pdf->SetMargins(PDF_MARGIN_LEFT, PDF_MARGIN_TOP, PDF_MARGIN_RIGHT);
        $pdf->SetMargins(PDF_MARGIN_LEFT, 10, PDF_MARGIN_RIGHT);
        $pdf->SetHeaderMargin(0);
        $pdf->SetFooterMargin(PDF_MARGIN_FOOTER);

        $pdf->footer = array(
            'bank_letter_title' => $voucher['bank_letter_title']
        );
        $pdf->lang = $lang;
        // set auto page breaks
        $pdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM);

        // add a page
        $pdf->AddPage();
        $pdf->SetFont('freesans','',10);
        $pdf->Cell(30,7,stdDate($voucher['bank_letter_date']));
        $pdf->ln(7);
        $pdf->Cell(60,7,$company_account['bank_name'].',');
        $pdf->ln(7);
        $pdf->Cell(60,7,$company_account['bank_branch']);
        $pdf->ln(7);
        $pdf->Cell(60,7,'Letter Ref: '.$voucher['document_identity']);
        $pdf->ln(14);
        $pdf->Cell(30,7,'Dear Sir,');
        $pdf->ln(7);
        $pdf->SetFont('freesans','BU',10);
        $pdf->Cell(15,7,'');
        $pdf->Cell(150,7,'Transfer from our Account# '.$company_account['account_no'].' ('.$company_account['account_title'].')',0,0,'L');
        $pdf->ln(15);
        $pdf->SetFont('freesans','',10);
        $pdf->Cell(150,7,'Please Transfer Rs. '.number_format($voucher['total_amount'],2).' ('.Number2Words(round($voucher['total_amount'])).')');
        $pdf->ln(7);
        $pdf->Cell(150,7,'in following account');
        $pdf->ln(14);
        $pdf->SetFont('freesans','B',7);
        $pdf->Cell(20,7,'SNo.','TB','','C');
        $pdf->Cell(70,7,'Account Title','TB','','L');
        $pdf->Cell(25,7,'Account No.','TB','','R');
        $pdf->Cell(30,7,'Rupees','TB','','R');
        $pdf->Cell(35,7,'Reference','TB','','R');
        $pdf->SetFont('freesans','',7);
        $sr_no=0;
        foreach($details as $detail) {
            $sr_no++;
            $pdf->ln(7);
            $pdf->Cell(20,7,$sr_no,'B','','C');
            $pdf->Cell(70,7,$detail['bank_title'],'B','','L');
            $pdf->Cell(25,7,$detail['bank_iban'],'B','','R');
            $pdf->Cell(30,7,number_format($detail['amount'],2),'B','','R');
            $pdf->Cell(35,5,$voucher['bank_reference'],'','','L');
        }
        $pdf->ln(14);
        $pdf->Cell(90,7,'','','','R');
        $pdf->Cell(25,7,'Total Rupees:','TB','','R');
        $pdf->Cell(30,7,number_format($voucher['total_amount'],2),'TB','','R');
        $pdf->ln(8);
        $pdf->Cell(80,7,'','','','R');
        $pdf->Cell(25,7,'Yours truly','T','','LB');
        $pdf->Cell(30,7,'','T','','R');
        $pdf->ln(20);
        $pdf->Cell(125,7,'','','','R');
        if($this->session->data['parent_company_id'] != 0) {
            $pdf->Cell(55,7,'For DAWAT-E-HADIYAH','','','L');
        } else {
            $pdf->Cell(55,7,$this->session->data['company_name'],'','','R');
        }

        $pdf->Output('Bank Letter:'.$voucher['bank_letter_no'].'@'.date('YmdHis').'.pdf', 'I');
    }

    public function printRTGS() {
        $bank_letter_id = $this->request->get['bank_letter_id'];
        $this->model['company_account'] = $this->load->model('setup/company_account');

        $lang = $this->load->language('payroll/bank_letter');
        $this->model['bank_letter'] = $this->load->model('payroll/bank_letter');
        $this->model['bank_letter_detail'] = $this->load->model('payroll/bank_letter_detail');

        $voucher = $this->model['bank_letter']->getRow(array('bank_letter_id' => $bank_letter_id));
        $company_account = $this->model['company_account']->getRow(array('company_account_id'=>$voucher['company_account_id']));
        // d($voucher,true);

        $details = $this->model['bank_letter_detail']->getRTGSDetails($bank_letter_id);

        $pdf = new PDF('P', PDF_UNIT, 'A4', true, 'UTF-8', false);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-Bold.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-BoldItalic.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-ExtraBold.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-ExtraBoldItalic.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-Italic.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-Light.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-LightItalic.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-Regular.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-SemiBold.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-SemiBoldItalic.ttf', 'TrueTypeUnicode', '', 96);
        // set document information
        $pdf->SetAuthor('Huzaifa Khambaty');
        $pdf->SetTitle('Bank Letter:'.$voucher['bank_letter_no'].'@'.date('YmdHis'));
        $pdf->SetSubject('Bank Letter');

        // set margins
        //$pdf->SetMargins(PDF_MARGIN_LEFT, PDF_MARGIN_TOP, PDF_MARGIN_RIGHT);
        $pdf->SetMargins(PDF_MARGIN_LEFT, 10, PDF_MARGIN_RIGHT);
        $pdf->SetHeaderMargin(0);
        $pdf->SetFooterMargin(PDF_MARGIN_FOOTER);

        $pdf->footer = array(
            'bank_letter_title' => $voucher['bank_letter_title']
        );
        $pdf->lang = $lang;
        // set auto page breaks
        $pdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM);

        // add a page
        $pdf->AddPage();
        $pdf->SetFont('freesans','',10);
        $pdf->Cell(30,7,'Date:'.stdDate($voucher['bank_letter_date']));
        $pdf->ln(7);
        $pdf->Cell(30,7,'The Manager,');
        $pdf->ln(7);
        $pdf->Cell(60,7,$company_account['bank_name'].',');
        $pdf->ln(7);
        $pdf->Cell(60,7,$company_account['bank_branch']);
        $pdf->ln(7);
        $pdf->Cell(60,7,'Letter Ref: '.$voucher['document_identity']);
        $pdf->ln(14);
        $pdf->SetFont('freesans','',8);
        $pdf->Cell(30,7,'Dear Sir,');
        $pdf->ln(7);
        $pdf->SetFont('freesans','B',8);
        $pdf->Cell(30,7,'Sub: Direct Transfer from Our Account No. '.$company_account['account_no']);
        $pdf->ln(7);
        $pdf->SetFont('freesans','',8);
        $pdf->Cell(180,7,'You are requested to please make direct transfer from our Account# '.$company_account['account_no'].' held in your branch to the Beneficiaries listed below as per the details');
        $pdf->ln(5);
        $pdf->Cell(180,7,'mentioned herein:');
        $pdf->ln(14);
        $pdf->SetFont('freesans','B',8);
        $pdf->Cell(10,7,'SNo.','TB','','C');
        $pdf->Cell(60,7,'Account Title','TB','','L');
        $pdf->Cell(35,7,'Bank Name','TB','','L');
        $pdf->Cell(35,7,'IBAN No.','TB','','L');
        $pdf->Cell(25,7,'Amount(PKR)','TB','','R');
        $pdf->Cell(25,7,'Reference','TB','','L');
        $pdf->SetFont('freesans','',7);
        $sr_no=0;
        foreach($details as $detail) {
            $sr_no++;
            $pdf->ln(7);
            $pdf->Cell(10,5,$sr_no,'','','C');
            $pdf->Cell(60,5,$detail['bank_title'],'','','L');
            $pdf->Cell(35,5,$detail['bank_name'],'','','L');
            $pdf->Cell(35,5,$detail['bank_iban'],'','','L');
            $pdf->Cell(25,5,number_format($detail['amount'],2),'','','R');
            $pdf->Cell(25,5,$voucher['bank_reference'],'','','L');
            $pdf->ln(5);
            $pdf->Cell(10,5,'','B','','C');
            $pdf->Cell(60,5,'('.$detail['cnic_no'].')','B','','L');
            $pdf->Cell(35,5,'','B','','R');
            $pdf->Cell(35,5,'','B','','R');
            $pdf->Cell(25,5,'','B','','R');
            $pdf->Cell(25,5,'','B','','R');
        }
        $pdf->ln(9);
        $pdf->SetFont('freesans','B',8);
        $pdf->Cell(10,7,'Total:','','','L');
        $pdf->Cell(160,7,Number2Words(round($voucher['total_amount'],0)).' only','','','L');
        $pdf->Cell(20,7,number_format($voucher['total_amount'],2),'TB','','R');
        $pdf->ln(9);
        $pdf->Cell(0,7,'Note: (Reference is to be mentioned on respective account is / are compulsory)','','','L');
        $pdf->ln(5);
        $pdf->Cell(0,7,'Payment be made through Transfer / IBFT / RTGS','','','L');
        $pdf->ln(8);
        $pdf->SetFont('freesans','',8);
        $pdf->Cell(0,7,'You are requested to please confirm Title of Account with IBAN before executing the Transfer. If there is any discrepancy,','','','L');
        $pdf->ln(5);
        $pdf->Cell(0,7,'please hold the Transfer and inform us accordingly. Any Transfer to wrong account (where Title doesn`t match with the IBAN)','','','L');
        $pdf->ln(5);
        $pdf->Cell(0,7,'will be the sole responsibility of JS Bank, and Bank shall be liable to refund the amount to us.','','','L');
        $pdf->ln(10);
        $pdf->Cell(0,7,'You are further requested to please provide Reference Number of each Transfer after execution of the transaction.','','','L');
        $pdf->ln(10);
        $pdf->Cell(55,7,'Yours truly','','','LB');
        $pdf->ln(5);
        if($this->session->data['parent_company_id'] != 0) {
            $pdf->Cell(55,7,'DAWAT-E-HADIYAH','','','L');
        } else {
            $pdf->Cell(55,7,$this->session->data['company_name'],'','','R');
        }
        $pdf->ln(25);
        $pdf->Cell(55,7,'Authorized Signatory','T','','LB');

        $pdf->Output('Bank Letter:'.$voucher['bank_letter_no'].'@'.date('YmdHis').'.pdf', 'I');
    }

    public function getAccountsData() {
        $this->model['bank_letter'] = $this->load->model('payroll/bank_letter');

        $post = $this->request->post;
        $bank_letter_id = $this->request->get['bank_letter_id'];
        $pay_date = $post['pay_date'];

        if($post['source_type']=='payroll') {
            // Payroll have Multiple Credit Entries
            $account_details = $this->model['bank_letter']->getPostDataPayroll($bank_letter_id, $post['posting_type']);
        } elseif($post['source_type']=='sp_enayat') {
            $account_details = $this->model['bank_letter']->getPostDataSpecialEnayat($bank_letter_id, $post['posting_type']);
        } else {
            $account_details = $this->model['bank_letter']->getPostDataSpecialAllowance($bank_letter_id, $post['posting_type']);
        }

        //d($account_details, true);
        $ctrl_no = 1;
        $data = array();
        $budgets = array();
        foreach($account_details as $detail) {
            if($detail['PartyID']==35) {
                $detail['Party'] = 'Employee OLD Age Benefits Institution';
            } elseif($detail['PartyID']==105) {
                $detail['Party'] = 'National bank of Pakistan';
            } elseif($detail['PartyID']==158) {
                $detail['Party'] = 'Yourself';
            } elseif($detail['PartyID']==28) {
                $detail['Party'] = 'National Bank of Pakistan';
            }
            if($detail['tran_type']=='D') {
                $budgets[] = array(
                    'Comp' => $post['Comp'],
                    'Year' => $detail['Year'],
                    'BudKey' => $detail['budget_key'],
                    'Ref' => str_replace('/','-',$detail['SysKey']),
                    'budAmount' => $detail['Pamount'],
                    'Status' => 'F',
                    'PartyID' => $detail['PartyID'],
                    'AHKey' => $detail['AHKey'],
                );
                $TranCategory = $detail['TranCategory'];
            }

            $detail['TranCategory'] = $TranCategory;
            $detail['Comp'] = $post['Comp'];
            $detail['PayDate'] = date('Ymd',strtotime($pay_date));
            //$detail['PayDate'] = date('Ymd',strtotime($detail['PayDate']));
            $detail['SysKey'] = str_replace('/','-',$detail['SysKey']);
            $detail['Ctrl_No'] = $ctrl_no;

//            if(isset($data[$detail['AHKey'].'-'.$detail['PartyID']])) {
//                $data[$detail['AHKey'].'-'.$detail['PartyID']]['Pamount'] += $detail['Pamount'];
//                $data[$detail['AHKey'].'-'.$detail['PartyID']]['Bamount'] += $detail['Bamount'];
//            } else {
//                $data[$detail['AHKey'].'-'.$detail['PartyID']] = $detail;
//            }
            $data[] = $detail;
        }

        $json = [
            'success' => true,
            'budgets' => $budgets,
            'accounts' => $data
        ];

        echo json_encode($json, true);
        exit;
    }

    public function post() {
        parent::getForm();
        $bank_letter_id = $this->request->get['bank_letter_id'];

        if (($this->request->server['REQUEST_METHOD'] == 'POST')) {
            $this->db->beginTransaction();
            $id = $this->postData($this->request->post);
            $this->db->commit();

            //$this->session->data['success'] = $this->language->get('success_insert');
            $this->redirect($this->url->link($this->getAlias(), 'token=' . $this->session->data['token'], 'SSL'));
        }

        $this->data['lang'] = $this->load->language($this->getAlias());
        $this->model['bank_letter'] = $this->load->model('payroll/bank_letter');
        $this->model['company_account'] = $this->load->model('setup/company_account');
        $this->model['bank_letter_detail'] = $this->load->model('payroll/bank_letter_detail');
        $this->model['company'] = $this->load->model('setup/company');

        $result = $this->model['bank_letter']->getRow(['bank_letter_id' => $bank_letter_id]);
        $company_account = $this->model['company_account']->getRow(['company_account_id' => $result['company_account_id']]);
        $company = $this->model['company']->getRow(['company_id' => $company_account['company_id']]);

        //d([$result, $company_account], true);
        foreach($result as $field => $value) {
            if($field=='document_date') {
                $this->data[$field] = stdDate($value);
            } else {
                $this->data[$field] = $value;
            }
        }
        foreach($company_account as $field => $value) {
            if($field=='document_date') {
                $this->data[$field] = stdDate($value);
            } else {
                $this->data[$field] = $value;
            }
        }
        $this->data['pay_from_company_id'] = $company['ref_id'];

        $this->data['companies'] = $this->model['company']->getRows(array(), array('company_name'));
        $this->data['cheque_details'] = $this->model['bank_letter_detail']->getRows(['bank_letter_id' => $bank_letter_id]);

        $this->data['href_get_accounts_data'] = $this->url->link($this->getAlias() . '/getAccountsData', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $this->request->get[$this->getPrimaryKey()], 'SSL');
        $this->data['action_save'] = $this->url->link($this->getAlias() . '/post', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $this->request->get[$this->getPrimaryKey()], 'SSL');
        $this->template = $this->getAlias() . '_post.tpl';
        $this->response->setOutput($this->render());
    }

    private function postData($data) {
        $this->model['bank_letter'] = $this->load->model('payroll/bank_letter');
        //d($data, true);
        try {
            $file = fopen(DIR_ROOT ."BLPosting_".date('YmdHi').".txt","a+");
            fwrite($file,'Request:'.date('YmdHis')."\n");
            fwrite($file,json_encode(array('ledgers'=>$data['account_details'], 'budgets'=>$data['budgets'])));
            fclose($file);

            $client = new SoapClient(ACC_SERV_URL);
            $response = $client->Payroll(array('Payrolldata' => json_encode($data['account_details']), 'Payrollbuddata' => json_encode($data['budgets'])));
            //d($response, true);
            fwrite($file,'Response:'.date('YmdHis'));
            fwrite($file,json_encode($response)."\n");
            fclose($file);
            if($response->PayrollResult->string[0]=='Success') {
                $data = array(
                    'is_post' => 0,
                    'post_at' => date('Y-m-d H:i:s'),
                    'post_by_id' => $this->session->data['user_id'],
                    'bno' => $response->PayrollResult->string[1]
                );
                $this->model['bank_letter']->edit($this->getAlias(),$this->request->get[$this->getPrimaryKey()],$data);
                $this->session->data['success'] = 'Data Posted Successfully on BNO: '. $response->PayrollResult->string[1];
            } else {
                $this->session->data['error_warning'] = $response->PayrollResult->string[1];
            }
        } catch (Exception $e) {
            d($e, true);
            $this->session->data['error_warning'] = $e->getMessage();
        }
    }

    public function downloadExcel() {
        include_once(DIR_SYSTEM.'library/PHPExcel.php');
        include_once(DIR_SYSTEM.'library/PHPExcel/Classes/PHPExcel/Writer/Excel5.php');
        $bank_letter_id = $this->request->get['bank_letter_id'];

        $lang = $this->load->language('payroll/bank_letter');
        $this->model['bank_letter'] = $this->load->model('payroll/bank_letter');
        $this->model['bank_letter_detail'] = $this->load->model('payroll/bank_letter_detail');
        $this->model['company_account'] = $this->load->model('setup/company_account');

        $voucher = $this->model['bank_letter']->getRow(array('bank_letter_id' => $bank_letter_id));
        $rows = $this->model['bank_letter_detail']->getRows(array('bank_letter_id' => $bank_letter_id), array('employee_name'));
        $company_account = $this->model['company_account']->getRow(array('company_account_id'=>$voucher['company_account_id']));

        //d($rows, true);
        $objPHPExcel = new PHPExcel();
        $objPHPExcel->setActiveSheetIndex(0);
        $rowCount = 1;
        $objPHPExcel->getActiveSheet()->SetCellValue('A'.$rowCount, stdDate($voucher['bank_letter_date']));
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('A'.$rowCount, $company_account['bank_name']);
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('A'.$rowCount, $company_account['bank_branch']);
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('A'.$rowCount, 'Dear Sir,');
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('B'.$rowCount, 'We are enclosing herewith Cheque No. '.$voucher['cheque_no'].'  Dated: '.stdDate($voucher['cheque_date']));
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('B'.$rowCount, 'for Rs. '.number_format($voucher['cheque_amount'],2).' drawn on '.$company_account['bank_name'].' being '.$voucher['document_title'].'.');
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('B'.$rowCount, 'kindly credit the following accounts');
        $rowCount++;
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('A'.$rowCount, 'SNo.')->getStyle('A'.$rowCount)->getFont()->setBold( true );
        $objPHPExcel->getActiveSheet()->SetCellValue('B'.$rowCount, 'Account Title')->getStyle('B'.$rowCount)->getFont()->setBold( true );
        $objPHPExcel->getActiveSheet()->SetCellValue('C'.$rowCount, 'Account No.')->getStyle('C'.$rowCount)->getFont()->setBold( true );
        $objPHPExcel->getActiveSheet()->SetCellValue('D'.$rowCount, 'Rupees')->getStyle('D'.$rowCount)->getFont()->setBold( true );
        $rowCount++;
        $sr_no=0;
        foreach($rows as $detail){
            $sr_no++;
            $objPHPExcel->getActiveSheet()->SetCellValue('A'.$rowCount, $sr_no);
            $objPHPExcel->getActiveSheet()->SetCellValue('B'.$rowCount, $detail['bank_title']);
            $objPHPExcel->getActiveSheet()->SetCellValue('C'.$rowCount, $detail['bank_iban']);
            $objPHPExcel->getActiveSheet()->SetCellValue('D'.$rowCount, number_format($detail['amount'],2));
            $rowCount++;
        }
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('C'.$rowCount, 'Total Rupees:')->getStyle('C'.$rowCount)->getFont()->setBold( true );
        $objPHPExcel->getActiveSheet()->SetCellValue('D'.$rowCount, number_format($voucher['total_amount'],2));
        $rowCount++;
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('C'.$rowCount, 'Yours truly,')->getStyle('C'.$rowCount)->getFont()->setBold( true );
        $rowCount++;
        $rowCount++;
        $rowCount++;
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('C'.$rowCount, $this->session->data['company_name'])->getStyle('C'.$rowCount)->getFont()->setBold( true );

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="Bank_Letter.xlsx"');
        header('Cache-Control: max-age=0');
        //$objWriter = PHPExcel_IOFactory::createWriter($objPHPExcel, 'Excel5');
        $objWriter = new PHPExcel_Writer_Excel2007($objPHPExcel);
        //$objWriter->save('some_excel_file.xlsx');
        $objWriter->save('php://output');
        exit;
    }

    public function viewSMS() {
        parent::getForm();
        $bank_letter_id = $this->request->get['bank_letter_id'];
        $this->model['employee_registration'] = $this->load->model('employee/employee_registration');
        $this->model['bank_letter'] = $this->load->model('payroll/bank_letter');
        $this->data['lang'] = $this->load->language($this->getAlias());
        $this->data['employees'] = $this->model['bank_letter']->getEmployeeData($bank_letter_id);
        $this->data['href_send_sms'] = $this->url->link($this->getAlias() . '/sendSMS', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $this->request->get[$this->getPrimaryKey()], 'SSL');

        //d($this->data['employees'], true);
        $bl = $this->model['bank_letter']->getRow(['bank_letter_id' => $bank_letter_id]);
        if($bl['source_type']=='payroll') {
            $this->data['default_message'] = "Your ".$bl['document_title']." is deposited in your bank.";
        } elseif($bl['source_type']=='sp_enayat') {
            $this->data['default_message'] = "Your Enayat is deposited in your bank";
        } else {
            $this->data['default_message'] = "Your Allowance is deposited in your bank";
        }

        $this->model['core_setting'] = $this->load->model('setup/core_setting');
        $config = $this->model['core_setting']->getConfig(['company_id' => $this->session->data['company_id']]);
        $user_name = $config['sms_user_name'];
        $password = $config['sms_password'];
        $from_mask = $config['sms_from_mask'];

        if($user_name=='' || $password=='' || $from_mask=='') {
            $this->data['warning'] = "Please verify you have valid SMS Setting configured.";
        }
        $this->template = $this->getAlias() . '_sms.tpl';
        $this->response->setOutput($this->render());
    }

    public function sendSMS() {
        $post = $this->request->post;
        $employee = $post['employee'];
        $message = $post['message'];

        $this->model['core_setting'] = $this->load->model('setup/core_setting');
        $config = $this->model['core_setting']->getConfig(['company_id' => $this->session->data['company_id']]);
        $user_name = $config['sms_user_name'];
        $password = $config['sms_password'];
        $from_mask = $config['sms_from_mask'];

        //d($config, true);
        $mobile_no = str_replace("+92","0",str_replace(" ","",$employee['mobile_no']));
        //$mobile_no = '03332221175';
        $url = 'https://connect.jazzcmt.com/sendsms_url.html?Username='.$user_name.'&Password='.$password.'&From='.$from_mask.'&To='.$mobile_no.'&Message=' . urlencode($message);
        $response = file_get_contents($url);
        $json['success'] = true;
        $json['mobile_no'] = $mobile_no;
        $json['response'] = $response;
        echo json_encode($json);
        exit;
    }

    public function viewEmail() {
        parent::getForm();
        $bank_letter_id = $this->request->get['bank_letter_id'];
        $this->model['employee_registration'] = $this->load->model('employee/employee_registration');
        $this->model['bank_letter'] = $this->load->model('payroll/bank_letter');
        $this->data['lang'] = $this->load->language($this->getAlias());
        $this->data['employees'] = $this->model['bank_letter']->getEmployeeData($bank_letter_id);
        $this->data['href_send_email'] = $this->url->link($this->getAlias() . '/sendEmail', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $this->request->get[$this->getPrimaryKey()], 'SSL');

        //d($this->data['employees'], true);
        $bl = $this->model['bank_letter']->getRow(['bank_letter_id' => $bank_letter_id]);
        if($bl['source_type']=='payroll') {
            $this->data['default_message'] = "Your ".$bl['document_title']." is deposited in your bank.";
        } elseif($bl['source_type']=='sp_enayat') {
            $this->data['default_message'] = "Your Enayat is deposited in your bank";
        } else {
            $this->data['default_message'] = "Your ".$bl['source_type']." Allowance is deposited in your bank";
        }

        $this->model['core_setting'] = $this->load->model('setup/core_setting');
        $config = $this->model['core_setting']->getConfig(['company_id' => $this->session->data['company_id']]);
        $user_name = $config['sms_user_name'];
        $password = $config['sms_password'];
        $from_mask = $config['sms_from_mask'];

        if($user_name=='' || $password=='' || $from_mask=='') {
            $this->data['warning'] = "Please verify you have valid SMTP and SMS Setting configured.";
        }

        $this->template = $this->getAlias() . '_email.tpl';
        $this->response->setOutput($this->render());
    }

    public function sendEmail() {
        $post = $this->request->post;
        //d($post, true);
        $this->model['core_setting'] = $this->load->model('setup/core_setting');
        if($post['source_type']=='payroll') {
            $this->model['bank'] = $this->load->model('setup/bank');
            $this->model['payroll'] = $this->load->model('payroll/payroll');
            $this->model['payroll_allowance'] = $this->load->model('payroll/payroll_allowance');
            $this->model['payroll_deduction'] = $this->load->model('payroll/payroll_deduction');
            //d($post, true);
            $post['payroll_id'] = $post['source_table_id'];
            $arrBanks = $this->model['bank']->getArrays('bank_id','bank_name',array());

            $payroll = $this->model['payroll']->getRow(['payroll_id' => $post['payroll_id']]);
            //d($payroll, true);
            $allowances = $this->model['payroll_allowance']->getRows(array('payroll_id' => $payroll['payroll_id']));
            $deductions = $this->model['payroll_deduction']->getRows(array('payroll_id' => $payroll['payroll_id']));
            //d([$payroll, $allowances, $deductions], true);
            $html_body = '';
            $html_body .= 'Employee Code: '. $payroll['registration_code'].'<br />';
            $html_body .= 'Employee Name: '. $payroll['employee_name'].'<br />';
            $html_body .= 'Department Name: '. $payroll['department_name'].'<br />';
            $html_body .= 'Email: '. $payroll['email'].'<br />';
            $html_body .= 'Basic Salary: '. number_format($payroll['basic_salary'],2).'<br />';
            $gross_amount = $payroll['basic_salary'];
            if($payroll['house_rent_allowance'] != 0) {
                $html_body .= 'House Rent: '. number_format($payroll['house_rent_allowance'],2).'<br />';
                //$gross_amount += $payroll['house_rent_allowance'];
            }
            foreach($allowances as $allowance) {
                if($allowance['allowance_amount'] != 0 && $allowance['grossable']) {
                    $html_body .= $allowance['allowance_name'] . ': '. number_format($allowance['allowance_amount'],2).'<br />';
                    $gross_amount += $payroll['allowance_amount'];
                }
            }
            if($payroll['arrears'] != 0) {
                $html_body .= 'Arrears: '. number_format($payroll['arrears'],2).'<br />';
                $gross_amount += $payroll['arrears'];
            }
            if($payroll['manual_addition'] != 0) {
                $html_body .= 'Other Allowance: '. number_format($payroll['manual_addition'],2).'<br />';
                $gross_amount += $payroll['manual_addition'];
            }
            if($payroll['overtime'] != 0) {
                $html_body .= 'Overtime: '. number_format($payroll['overtime'],2).'<br />';
                $gross_amount += $payroll['overtime'];
            }
            $html_body .= '<b>Gross Amount:</b> '. number_format($gross_amount,2).'<br />';
            foreach($deductions as $deduction) {
                if($deduction['deduction_amount'] != 0) {
                    $html_body .= $deduction['deduction_name'] . ': '. number_format($deduction['deduction_amount'],2).'<br />';
                }
            }
            if($payroll['manual_deduction'] != 0) {
                $html_body .= 'Other Deduction: '. number_format($payroll['manual_deduction'],2).'<br />';
            }
            if($payroll['attendance_deduction'] != 0) {
                $html_body .= 'Absent Deduction: '. number_format($payroll['attendance_deduction'],2).'<br />';
            }
            if($payroll['tax_deduction'] != 0) {
                $html_body .= 'Tax: '. number_format($payroll['tax_deduction'],2).'<br />';
            }
            $html_body .= '<b>Net Salary: </b>'. number_format($payroll['payable_salary'],2).'<br />';
            $html_body .= 'Credited to A/C No. '. $payroll['bank_account_no'].' in '.$arrBanks[$payroll['bank_id']];
        } else {
            $html_body = '';
            $html_body .= 'Employee Code: '. $post['registration_code'].'<br />';
            $html_body .= 'Employee Name: '. $post['employee_name'].'<br />';
            $html_body .= 'Department Name: '. $post['department_name'].'<br />';
            $html_body .= 'Email: '. $post['email'].'<br />';
            $html_body .= 'Amount: '. number_format($post['amount'],2).'<br />';
            $html_body .= 'Credited to A/C No. '. $post['bank_iban'].' in '.$post['bank_name'];
        }


        $config = $this->model['core_setting']->getArrays('field','value',array('company_id' => $this->session->data['company_id'], 'module'=>'general'));
        //d($config, true);
        try {
            $mail = new PHPMailer(true);                              // Passing `true` enables exceptions
            $mail->SMTPOptions = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );
            //Server settings
            $mail->SMTPDebug = 0;                                 // Enable verbose debug output
            $mail->isSMTP();                                      // Set mailer to use SMTP
            $mail->Host = $config['config_mail_smtp_hostname'];
            $mail->SMTPAuth = true;
            $mail->Username = $config['config_mail_smtp_username']; //paste one generated by Mailtrap
            $mail->Password = $config['config_mail_smtp_password']; //paste one generated by Mailtrap
            //$mail->SMTPSecure = 'tls';
            $mail->Port = $config['config_mail_smtp_port'];
            $mail->setFrom($config['config_mail_smtp_username'], $post['config_mail_smtp_displayname']);
            $mail->addAddress($post['email']);
            //$mail->addAddress('huzaifa.khambaty@gmail.com');
            $mail->isHTML(true);
            // Set email format to HTML
            $mail->Subject = $post['document_title'];
            $mail->Body    = $html_body;
            if(!$mail->Send()) {
                $json = [
                    'success' => false,
                    'error' => $mail->ErrorInfo,
                    'id' => $post['id'],
                    'next_id' => $post['id']+1,
                ];
            } else {
                $json = [
                    'success' => true,
                    'id' => $post['id'],
                    'next_id' => $post['id']+1,
                ];
            }
        } catch (Exception $e) {
            $json = [
                'success' => false,
                'error' => $e->getCode().'|'.$e->getMessage(),
                'id' => $post['id'],
                'next_id' => $post['id']+1,
            ];
        }


        echo json_encode($json);
    }

    public function customExcel() {
        $this->init();
        parent::getForm();

        $this->data['href_save_report'] = $this->url->link($this->getAlias() . '/saveCustomFormat', 'token=' . $this->session->data['token'], 'SSL');
        $this->data['href_load_formats'] = $this->url->link($this->getAlias() . '/loadCustomFormats', 'token=' . $this->session->data['token'], 'SSL');
        $this->data['href_remove_format'] = $this->url->link($this->getAlias() . '/removeCustomFormat', 'token=' . $this->session->data['token'], 'SSL');

        $columns = array();
        $columns[] = array(
            'column_ratio' => 1,
            'column_name' => 's_no',
            'display_name' => 'Sr.'
        );
        $columns[] = array(
            'column_ratio' => 4,
            'column_name' => 'dp.department_name',
            'display_name' => 'Department'
        );
        $columns[] = array(
            'column_ratio' => 4,
            'column_name' => 'd.designation_name',
            'display_name' => 'Designation'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'e.employee_code',
            'display_name' => 'Emp. Code'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'er.registration_code',
            'display_name' => 'Reg. Code'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'er.attendance_code',
            'display_name' => 'Att. Code'
        );
        $columns[] = array(
            'column_ratio' => 4,
            'column_name' => 'e.full_name',
            'display_name' => 'Full Name'
        );
        $columns[] = array(
            'column_ratio' => 4,
            'column_name' => 'ean.name_ar',
            'display_name' => 'Arabic Name'
        );
        $columns[] = array(
            'column_ratio' => 3,
            'column_name' => 'e.surname',
            'display_name' => 'Surname'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'e.its_no',
            'display_name' => 'ITS No.'
        );
//        $columns[] = array(
//            'column_ratio' => 3,
//            'column_name' => 'sh.shift_title',
//            'display_name' => 'Shift'
//        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'e.dob',
            'display_name' => 'DOB'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'e.gender',
            'display_name' => 'Gender'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'e.marital_status',
            'display_name' => 'Marital Status'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'e.mobile_no',
            'display_name' => 'Mobile No.'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'e.phone_no',
            'display_name' => 'Phone No.'
        );
        $columns[] = array(
            'column_ratio' => 3,
            'column_name' => 'e.email',
            'display_name' => 'Email'
        );
        $columns[] = array(
            'column_ratio' => 3,
            'column_name' => 'e.cnic_no',
            'display_name' => 'CNIC NO.'
        );
        $columns[] = array(
            'column_ratio' => 4,
            'column_name' => 'e.cnic_name',
            'display_name' => 'CNIC Name'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'e.cnic_expiry',
            'display_name' => 'CNIC Expiry'
        );
        $columns[] = array(
            'column_ratio' => 6,
            'column_name' => 'e.cnic_address',
            'display_name' => 'CNIC Address'
        );
        $columns[] = array(
            'column_ratio' => 1,
            'column_name' => 'e.eobi_no',
            'display_name' => 'EOBI No.'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'e.eobi_date',
            'display_name' => 'EOBI Date'
        );
        $columns[] = array(
            'column_ratio' => 1,
            'column_name' => 'e.eobi_sno',
            'display_name' => 'EOBI SNo.'
        );
        $columns[] = array(
            'column_ratio' => 1,
            'column_name' => 'e.nicop_no',
            'display_name' => 'NICOP No.'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'e.nicop_name',
            'display_name' => 'NICOP Name'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'e.nicop_expiry',
            'display_name' => 'NICOP Expiry'
        );
        $columns[] = array(
            'column_ratio' => 4,
            'column_name' => 'e.nicop_address',
            'display_name' => 'NICOP Address'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'e.passport_no',
            'display_name' => 'Passport No.'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'e.passport_name',
            'display_name' => 'Passport Name'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'e.passport_expiry',
            'display_name' => 'Passport Expiry'
        );
        $columns[] = array(
            'column_ratio' => 3,
            'column_name' => 'e.passport_address',
            'display_name' => 'Passport Address'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'bl.document_date',
            'display_name' => 'Date'
        );
        $columns[] = array(
            'column_ratio' => 3,
            'column_name' => 'bl.document_identity',
            'display_name' => 'Doc. No.'
        );
        $columns[] = array(
            'column_ratio' => 4,
            'column_name' => 'bld.bank_name',
            'display_name' => 'Bank Name'
        );
        $columns[] = array(
            'column_ratio' => 4,
            'column_name' => 'bnk.bank_code',
            'display_name' => 'Bank Code'
        );
        $columns[] = array(
            'column_ratio' => 4,
            'column_name' => 'bld.bank_title',
            'display_name' => 'Bank Title'
        );
        $columns[] = array(
            'column_ratio' => 4,
            'column_name' => 'bld.bank_iban',
            'display_name' => 'Bank IBAN'
        );
        $columns[] = array(
            'column_ratio' => 2,
            'column_name' => 'bld.amount',
            'display_name' => 'Amount'
        );
        $this->data['columns'] = $columns;


        $this->template = $this->getAlias() . '_custom_excel.tpl';
        $this->response->setOutput($this->render());
    }

    public function saveCustomFormat() {
        $data = $this->request->post;
        $data['company_id'] = $this->session->data['company_id'];
        $data['report_columns'] = json_encode($data['display_columns']);

        $this->model['bank_letter_custom_format'] = $this->load->model('payroll/bank_letter_custom_format');
        if(isset($data['report_id']) && $data['report_id'] != "") {
            $report_id = $this->model['bank_letter_custom_format']->edit($this->getAlias(), $data['report_id'], $data);
        } else {
            $report_id = $this->model['bank_letter_custom_format']->add($this->getAlias(), $data);
        }
        echo json_encode(['success' => true, 'report_id' => $report_id]);
        exit;
    }

    public function loadCustomFormats() {
        $this->model['bank_letter_custom_format'] = $this->load->model('payroll/bank_letter_custom_format');

        $records = $this->model['bank_letter_custom_format']->getDistinctRows(['id','report_title','report_columns'],['company_id' => $this->session->data['company_id']]);
        $data = [];
        foreach($records as $record) {
            $data[$record['id']] = [
                'report_id' => $record['id'],
                'report_title' => $record['report_title'],
                'display_columns' => json_decode($record['report_columns'], true),
            ];
        }
        echo json_encode(['success' => true, 'data'=>$data]);
        exit;
    }

    public function removeCustomFormat() {
        $report_id = $this->request->get['report_id'];
        $this->model['bank_letter_custom_format'] = $this->load->model('payroll/bank_letter_custom_format');
        $this->model['bank_letter_custom_format']->delete($this->getAlias(), $report_id);
        echo json_encode(['success' => true, 'report_id' => $report_id]);
        exit;
    }

    public function downloadCustomExcel() {
        $session = $this->session->data;
        $get = $this->request->get;
        $bank_letter_id = $this->request->get['bank_letter_id'];
        $format_id = $this->request->get['format_id'];

        $this->model['bank_letter_custom_format'] = $this->load->model('payroll/bank_letter_custom_format');
        $this->model['bank_letter_detail'] = $this->load->model('payroll/bank_letter_detail');
        $this->model['bank_letter'] = $this->load->model('payroll/bank_letter');

        $format = $this->model['bank_letter_custom_format']->getRow(['id' => $format_id]);
        $columns = json_decode($format['report_columns'],true);
        $arrColumns = [];
        foreach($columns as $idx => $column) {
            if($column['column_name']=='s_no') {
                $arrColumns[] = "@rownum:=@rownum+1 as '".$column['display_name']."'";
            } else {
                $arrColumns[] = $column['column_name']." as '".$column['display_name']."'";
            }

            if($column['column_name']=='bld.amount') {
                $column_no_for_total = $idx;
            }
        }
        $rows = $this->model['bank_letter_detail']->getColumnRecords($arrColumns,['bl.bank_letter_id' => $bank_letter_id]);
        $display_columns = array_keys($rows[0]);
        $this->model['company_account'] = $this->load->model('setup/company_account');

        $voucher = $this->model['bank_letter']->getRow(array('bank_letter_id' => $bank_letter_id));
        $company_account = $this->model['company_account']->getRow(array('company_account_id'=>$voucher['company_account_id']));

        $objPHPExcel = new PHPExcel();
        $objPHPExcel->setActiveSheetIndex(0);
        $rowCount = 1;
        $objPHPExcel->getActiveSheet()->SetCellValue('A'.$rowCount, stdDate($voucher['bank_letter_date']));
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('A'.$rowCount, $company_account['bank_name']);
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('A'.$rowCount, $company_account['bank_branch']);
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('A'.$rowCount, 'Dear Sir,');
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('B'.$rowCount, 'We are enclosing herewith Cheque No. '.$voucher['cheque_no'].'  Dated: '.stdDate($voucher['cheque_date']));
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('B'.$rowCount, 'for Rs. '.number_format($voucher['total_amount'],2).' drawn on '.$company_account['bank_name'].' being '.$voucher['document_title'].'.');
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('B'.$rowCount, 'kindly credit the following accounts');
        $rowCount++;
        $rowCount++;
        $column_no = 0;
        foreach($display_columns as $column) {
            $objPHPExcel->getActiveSheet()
                ->setCellValueByColumnAndRow($column_no, $rowCount, $column)
                ->getStyleByColumnAndRow($column_no, $rowCount)
                ->applyFromArray(
                    array(
                        'fill' => array(
                            'type' => PHPExcel_Style_Fill::FILL_SOLID,
                            'color' => array('rgb' => '000000')
                        )
                    ,
                        'font'  => array(
                            'color' => array('rgb' => 'ffffff')
                        )
                    )
                )
            ;
            $column_no++;
        }

        $rowCount++;
        foreach($rows as $detail){
            $column_no = 0;
            foreach($display_columns as $column) {
                $objPHPExcel->getActiveSheet()
                    ->setCellValueByColumnAndRow($column_no, $rowCount, $detail[$column])
                    ->getStyleByColumnAndRow($column_no, $rowCount)
//                    ->applyFromArray(
//                        array(
//                            'fill' => array(
//                                'type' => PHPExcel_Style_Fill::FILL_SOLID,
//                                'color' => array('rgb' => '000000')
//                            )
//                        ,
//                            'font'  => array(
//                                'color' => array('rgb' => 'ffffff')
//                            )
//                        )
//                    )
                ;
                $column_no++;
            }
            $rowCount++;
        }
        $rowCount++;
        if($column_no_for_total==0) {
            $objPHPExcel->getActiveSheet()
                ->setCellValueByColumnAndRow($column_no_for_total, $rowCount, number_format($voucher['total_amount'],2))
                ->getStyleByColumnAndRow($column_no_for_total, $rowCount)
                ->getFont()->setBold( true )
//                ->applyFromArray(
//                    array(
//                        'fill' => array(
//                            'type' => PHPExcel_Style_Fill::FILL_SOLID,
//                            'color' => array('rgb' => '000000')
//                        )
//                    ,
//                        'font'  => array(
//                            'color' => array('rgb' => 'ffffff')
//                        )
//                    )
//                )
            ;
        } else {
            $objPHPExcel->getActiveSheet()
                ->setCellValueByColumnAndRow($column_no_for_total-1, $rowCount, 'Total')
                ->getStyleByColumnAndRow($column_no_for_total-1, $rowCount)
                ->getFont()->setBold( true )
//                ->applyFromArray(
//                    array(
//                        'fill' => array(
//                            'type' => PHPExcel_Style_Fill::FILL_SOLID,
//                            'color' => array('rgb' => '000000')
//                        )
//                    ,
//                        'font'  => array(
//                            'color' => array('rgb' => 'ffffff')
//                        )
//                    )
//                )
            ;
            $objPHPExcel->getActiveSheet()
                ->setCellValueByColumnAndRow($column_no_for_total, $rowCount, number_format($voucher['total_amount'],2))
                ->getStyleByColumnAndRow($column_no_for_total, $rowCount)
                ->getFont()->setBold( true )
//                ->applyFromArray(
//                    array(
//                        'fill' => array(
//                            'type' => PHPExcel_Style_Fill::FILL_SOLID,
//                            'color' => array('rgb' => '000000')
//                        )
//                    ,
//                        'font'  => array(
//                            'color' => array('rgb' => 'ffffff')
//                        )
//                    )
//                )
            ;
        }
        $rowCount++;
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('B'.$rowCount, 'Yours truly,')->getStyle('C'.$rowCount)->getFont()->setBold( true );
        $rowCount++;
        $rowCount++;
        $rowCount++;
        $rowCount++;
        $objPHPExcel->getActiveSheet()->SetCellValue('B'.$rowCount, $this->session->data['company_name'])->getStyle('C'.$rowCount)->getFont()->setBold( true );

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="Bank_Letter.xlsx"');
        header('Cache-Control: max-age=0');
        //$objWriter = PHPExcel_IOFactory::createWriter($objPHPExcel, 'Excel5');
        $objWriter = new PHPExcel_Writer_Excel2007($objPHPExcel);
        //$objWriter->save('some_excel_file.xlsx');
        $objWriter->save('php://output');
        exit;
    }
}

class PDF extends TCPDF {
    public $lang;
    public $header;
    public $footer;
    //Page header
    public function Header() {

    }

    // Page footer
    public function Footer() {
        $data = $this->footer;
        // Position at 15 mm from bottom
        $this->SetY(-15);
        // Set font
        $this->SetFont('freesans', 'I', 8);
        // Page number
        $this->Cell(60, 10, $this->lang['ref'].': '.$data['document_title'], 0, false, 'L', 0, '', 0, false, 'T', 'M');
        $this->Cell(60, 10, 'Page '.$this->getAliasNumPage().'/'.$this->getAliasNbPages(), 0, false, 'C', 0, '', 0, false, 'T', 'M');
        $this->Cell(60, 10, 'Date: '.date('d-m-Y H:i'), 0, false, 'R', 0, '', 0, false, 'T', 'M');
    }
}
?>