<?php
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

require DIR_SYSTEM . 'library/PHPMailer/Exception.php';
require DIR_SYSTEM . 'library/PHPMailer/PHPMailer.php';
require DIR_SYSTEM . 'library/PHPMailer/SMTP.php';

class ControllerPayrollIndividualCheque extends HController {

    protected function getAlias() {
        return 'payroll/individual_cheque';
    }

    protected function getPrimaryKey() {
        return 'individual_cheque_id';
    }

    protected function getList() {
        parent::getList();

        $this->data['action_ajax'] = $this->url->link($this->getAlias() . '/getAjaxLists', 'token=' . $this->session->data['token'], 'SSL');
        $this->response->setOutput($this->render());
    }

    public function getAjaxLists() {

        $lang = $this->load->language($this->getAlias());
        $this->model[$this->getAlias()] = $this->load->model($this->getAlias());
        $data = array();
        $aColumns = array('action', 'document_date', 'document_identity', 'document_title', 'total_employee','total_amount','created_at','check_box');

        /*
         * Paging
         */
        $sLimit = "";
        if (isset($_GET['iDisplayStart']) && $_GET['iDisplayLength'] != '-1') {
            $data['criteria']['start'] = $_GET['iDisplayStart'];
            $data['criteria']['limit'] = $_GET['iDisplayLength'];
        }

        /*
         * Ordering
         */
        $sOrder = "";
        if (isset($_GET['iSortCol_0'])) {
            $sOrder = " ORDER BY  ";
            for ($i = 0; $i < intval($_GET['iSortingCols']); $i++) {
                if ($_GET['bSortable_' . intval($_GET['iSortCol_' . $i])] == "true") {
                    $sOrder .= "`" . $aColumns[intval($_GET['iSortCol_' . $i])] . "` " .
                        ($_GET['sSortDir_' . $i] === 'asc' ? 'asc' : 'desc') . ", ";
                }
            }

            $sOrder = substr_replace($sOrder, "", -2);
            if ($sOrder == " ORDER BY") {
                $sOrder = "";
            }
            $data['criteria']['orderby'] = $sOrder;
        }


        /*
         * Filtering
         * NOTE this does not match the built-in DataTables filtering which does it
         * word by word on any field. It's possible to do here, but concerned about efficiency
         * on very large tables, and MySQL's regex functionality is very limited
         */
        $arrWhere = array();
        $arrWhere[] = "`company_id` = '".$this->session->data['company_id']."'";
        $arrWhere[] = "`fiscal_year_id` = '".$this->session->data['fiscal_year_id']."'";
        if (isset($_GET['sSearch']) && $_GET['sSearch'] != "") {
            $arrSSearch = array();
            for ($i = 0; $i < count($aColumns); $i++) {
                if (isset($_GET['bSearchable_' . $i]) && $_GET['bSearchable_' . $i] == "true" && $_GET['sSearch'] != '') {
                    $arrSSearch[] = "LOWER(`" . $aColumns[$i] . "`) LIKE '%" . $this->db->escape(strtolower($_GET['sSearch'])) . "%'";
                }
            }
            if(!empty($arrSSearch)) {
                $arrWhere[] = '(' . implode(' OR ', $arrSSearch) . ')';
            }
        }

        /* Individual column filtering */
        for ($i = 0; $i < count($aColumns); $i++) {
            if (isset($_GET['bSearchable_' . $i]) && $_GET['bSearchable_' . $i] == "true" && $_GET['sSearch_' . $i] != '') {
                $arrWhere[] = "LOWER(`" . $aColumns[$i] . "`) LIKE '%" . $this->db->escape(strtolower($_GET['sSearch_' . $i])) . "%' ";
            }
        }

        if (!empty($arrWhere)) {
            //$data['filter']['RAW'] = substr($sWhere, 5, strlen($sWhere) - 5);
            $data['filter']['RAW'] = implode(' AND ', $arrWhere);
        }

        //d($data, true);
        $results = $this->model[$this->getAlias()]->getLists($data);
        $iFilteredTotal = $results['total'];
        $iTotal = $results['table_total'];


        /*
         * Output
         */
        $output = array(
            "sEcho" => intval($_GET['sEcho']),
            "iTotalRecords" => $iTotal,
            "iTotalDisplayRecords" => $iFilteredTotal,
            "aaData" => array()
        );

        foreach ($results['lists'] as $aRow) {
            $row = array();
            $actions = array();

            $actions[] = array(
                'text' => $lang['edit'],
                'href' => $this->url->link($this->getAlias() . '/update', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $aRow[$this->getPrimaryKey()], 'SSL'),
                'btn_class' => 'btn btn-primary btn-xs',
                'class' => 'fa fa-pencil'
            );
            $actions[] = array(
                'text' => $this->data['lang']['print'],
                'target' => '_blank',
                'href' => $this->url->link($this->getAlias() . '/printDocument', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $aRow[$this->getPrimaryKey()], 'SSL'),
                'btn_class' => 'btn btn-info btn-xs',
                'class' => 'fa fa-print'
            );
            $actions[] = array(
                'text' => $lang['sms'],
                'target' => '_blank',
                'href' => $this->url->link($this->getAlias() . '/viewSMS', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $aRow[$this->getPrimaryKey()], 'SSL'),
                'btn_class' => 'btn btn-xs btn-success',
                'class' => 'fa fa-commenting-o'
            );

            $actions[] = array(
                'text' => $lang['email'],
                'target' => '_blank',
                'href' => $this->url->link($this->getAlias() . '/viewEmail', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $aRow[$this->getPrimaryKey()], 'SSL'),
                'btn_class' => 'btn btn-xs btn-success',
                'class' => 'fa fa-at'
            );

            if($aRow['is_post'] != 1) {
                $actions[] = array(
                    'text' => $lang['post'],
                    'target' => '_blank',
                    'href' => $this->url->link($this->getAlias() . '/post', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $aRow[$this->getPrimaryKey()], 'SSL'),
                    'btn_class' => 'btn btn-info btn-xs',
                    'class' => 'fa fa-thumbs-up'
                );

                $actions[] = array(
                    'text' => $lang['delete'],
                    'href' => 'javascript:void(0);',
                    'click' => "ConfirmDelete('" . $this->url->link($this->getAlias() . '/delete', 'token=' . $this->session->data['token'] . '&id=' . $aRow[$this->getPrimaryKey()], 'SSL') . "')",
                    'btn_class' => 'btn btn-danger btn-xs',
                    'class' => 'fa fa-times'
                );
            }

            $strAction = '';
            foreach ($actions as $action) {
                $strAction .= '<a '.(isset($action['target'])?'target="'.$action['target'].'"':'').' '.(isset($action['btn_class'])?'class="'.$action['btn_class'].'"':'').' href="' . $action['href'] . '" data-toggle="tooltip" title="' . $action['text'] . '" ' . (isset($action['click']) ? 'onClick="' . $action['click'] . '"' : '') . '>';
                if (isset($action['class'])) {
                    $strAction .= '<span class="' . $action['class'] . '"></span>';
                } else {
                    $strAction .= $action['text'];
                }
                $strAction .= '</a>&nbsp;';
            }

            for ($i = 0; $i < count($aColumns); $i++) {
                if ($aColumns[$i] == 'action') {
                    $row[] = $strAction;
                } elseif ($aColumns[$i] == 'created_at') {
                    $row[] = stdDateTime($aRow['created_at']);
                } elseif ($aColumns[$i] == 'individual_cheque_date') {
                    $row[] = stdDate($aRow['individual_cheque_date']);
                } elseif ($aColumns[$i] == 'cheque_date') {
                    $row[] = stdDate($aRow['cheque_date']);
                } else {
                    $row[] = $aRow[$aColumns[$i]];
                }

            }
            $output['aaData'][] = $row;
        }

        echo json_encode($output);
    }

    protected function getForm() {
        parent::getForm();

        $this->model['individual_cheque'] = $this->load->model('payroll/individual_cheque');
        $this->model['department'] = $this->load->model('setup/department');
        $this->model['employee_registration'] = $this->load->model('employee/employee_registration');
        $this->model['company_account'] = $this->load->model('setup/company_account');
        $this->model['allowance_type'] = $this->load->model('setup/special_allowance_type');
        $this->model['user_access'] = $this->load->model('user/user_access');

        $company_ids = $this->model['user_access']->getArrays('company_id','company_id',array('user_id'=> $this->session->data['user_id']));
        $allowance_types = $this->model['allowance_type']->getRows(array(), array('special_allowance_type'));

        $source_types = array(
            'payroll' => 'Payroll',
            'sp_enayat' => 'Special Enayat',
        );
        foreach($allowance_types as $allowance_type) {
            $source_types[$allowance_type['special_allowance_type_id']] = $allowance_type['special_allowance_type'];
        }


        $this->data['source_types']  = $source_types;
        $this->data['employee_banks'] = $this->model['individual_cheque']->getEmployeeBanks();
        $this->data['departments'] = $this->model['department']->getRows(array('company_id' => $this->session->data['company_id']), array('department_name'));
        $this->data['employees'] = $this->model['employee_registration']->getRows(array('company_id' => $this->session->data['company_id']), array('employee_name'));

        $filter = [];
        if($this->session->data['is_all_company_account']!=1)
            $filter['company_id'] = $this->session->data['company_id'];

        $this->data['company_accounts'] = $this->model['company_account']->getRows($filter, array('account_title'));

        $this->data['document_date'] = stdDate();
        $this->data['is_edit'] = 0;
        if (isset($this->request->get['individual_cheque_id']) && ($this->request->server['REQUEST_METHOD'] != 'POST')) {
            $this->data['is_edit'] = 1;
            $result = $this->model[$this->getAlias()]->getRow(array('individual_cheque_id' => $this->request->get['individual_cheque_id']));
            foreach($result as $field => $value) {
                if(($field=='document_date' ||  $field=='source_date')  && $value != '') {
                    $this->data[$field] = stdDate($value);
                } else {
                    $this->data[$field] = $value;
                }
            }

            $this->model['individual_cheque_detail'] = $this->load->model('payroll/individual_cheque_detail');
            $details = $this->model['individual_cheque_detail']->getRows(array('individual_cheque_id' => $result['individual_cheque_id']));
            //d($details, true);
            foreach($details as $detail) {
                $detail['cheque_date'] = stdDate($detail['cheque_date']);

                $this->data['details'][] = $detail;
            }
        }

        $this->data['strValidation']="{
            'rules':{
		        'document_date': {'required':true},
		        'document_title': {'required':true},
		        'source_type': {'required':true},
		        'source_date': {'required':true},
		        'total_employee': {'required':true,'min':1},
		        'total_amount': {'required':true,'min':1},
            },
        }";

        $this->data['href_get_data'] = $this->url->link($this->getAlias() . '/getData', 'token=' . $this->session->data['token'] . '&individual_cheque_id=' . $this->request->get['individual_cheque_id']);
        $this->response->setOutput($this->render());
    }

    public function getData() {
        $post = $this->request->post;
        if($post['source_type']=='payroll') {
            $post['company_id'] = $this->session->data['company_id'];
            $post['fiscal_year_id'] = $this->session->data['fiscal_year_id'];
            $post['source_date'] = MySqlDate($post['source_date']);

            $this->model['individual_cheque'] = $this->load->model('payroll/individual_cheque');
            if($post['show_deduction']=='Yes') {
                $accounts = $this->model['individual_cheque']->getPayrollDeductions($post['source_date'], $post['company_id']);
            } else {
                $accounts = [];
            }
            $rows = $this->model['individual_cheque']->getPayrollData($post);
            $rows = array_merge($accounts, $rows);
        } elseif($post['source_type']=='sp_enayat') {
            $post['company_id'] = $this->session->data['company_id'];
            $post['fiscal_year_id'] = $this->session->data['fiscal_year_id'];
            $post['source_date'] = MySqlDate($post['source_date']);

            $this->model['individual_cheque'] = $this->load->model('payroll/individual_cheque');
            $rows = $this->model['individual_cheque']->getSpecialEnayatData($post);
        } else {
            $post['company_id'] = $this->session->data['company_id'];
            $post['fiscal_year_id'] = $this->session->data['fiscal_year_id'];
            $post['special_allowance_type_id'] = $post['source_type'];
            $post['source_date'] = MySqlDate($post['source_date']);

            $this->model['individual_cheque'] = $this->load->model('payroll/individual_cheque');
            $rows = $this->model['individual_cheque']->getSpecialAllowanceData($post);
        }

        $json = array(
            'success' => true,
            'employees' => $rows
        );

        echo json_encode($json);
    }

    protected function insertData($data) {
        $session = $this->session->data;
        $document_prefix = 'IC'.$session['fy_code'].'-';

        $document = $this->model[$this->getAlias()]->getNextNumber($document_prefix);
        $data['document_prefix'] = $document['document_prefix'];
        $data['document_no'] = $document['document_no'];
        $data['document_identity'] = $document['document_identity'];
        $data['document_date']=MySqlDate($data['document_date']);
        $data['company_id'] = $this->session->data['company_id'];
        $data['fiscal_year_id'] = $this->session->data['fiscal_year_id'];
        $data['source_date']=MySqlDate($data['source_date']);
        $data['department_id']=implode(',', $data['department_id']);

        $this->model['payroll'] = $this->load->model('payroll/payroll');

        $individual_cheque_id = $this->model[$this->getAlias()]->add($this->getAlias(), $data);
        $this->model['individual_cheque_detail'] = $this->load->model('payroll/individual_cheque_detail');
        foreach($data['employees'] as $detail) {
            $detail['individual_cheque_id'] = $individual_cheque_id;
            $detail['cheque_date'] = MySqlDate($detail['cheque_date']);

            $this->model['individual_cheque_detail']->add($this->getAlias(), $detail);
            $this->model['payroll']->edit($this->getAlias(), $detail['source_table_id'], ['deposit_slip_id' => $individual_cheque_id]);
        }

        return $individual_cheque_id;
    }

    protected function updateData($primary_key, $data) {
        $data['company_id'] = $this->session->data['company_id'];
        $data['fiscal_year_id'] = $this->session->data['fiscal_year_id'];
        $data['individual_cheque_date']=MySqlDate($data['individual_cheque_date']);
        $data['document_date']=MySqlDate($data['document_date']);
        $data['source_date']=MySqlDate($data['source_date']);

        $this->model['payroll'] = $this->load->model('payroll/payroll');

        $individual_cheque_id = $this->model[$this->getAlias()]->edit($this->getAlias(), $primary_key, $data);
        $this->model['individual_cheque_detail'] = $this->load->model('payroll/individual_cheque_detail');
        $this->model['individual_cheque_detail']->deleteBulk($this->getAlias(), array('individual_cheque_id' => $individual_cheque_id));

        foreach($data['employees'] as $detail) {
            $detail['individual_cheque_id'] = $individual_cheque_id;
            $detail['cheque_date'] = MySqlDate($detail['cheque_date']);

            $this->model['individual_cheque_detail']->add($this->getAlias(), $detail);
            $this->model['payroll']->edit($this->getAlias(), $detail['source_table_id'], ['deposit_slip_id' => $individual_cheque_id]);
        }

        return $individual_cheque_id;

    }

    protected function deleteData($primary_key) {
        $this->model['individual_cheque_detail'] = $this->load->model('payroll/individual_cheque_detail');
        $this->model['individual_cheque_detail']->deleteBulk($this->getAlias(), array('individual_cheque_id' => $primary_key));
        $this->model[$this->getAlias()]->delete($this->getAlias(), $primary_key);
    }

    public function printStatement() {
        $individual_cheque_id = $this->request->get['individual_cheque_id'];

        $lang = $this->load->language('payroll/individual_cheque');
        $this->model['individual_cheque'] = $this->load->model('payroll/individual_cheque');
        $this->model['individual_cheque_detail'] = $this->load->model('payroll/individual_cheque_detail');

        $voucher = $this->model['individual_cheque']->getRow(array('individual_cheque_id' => $individual_cheque_id));
        $details = $this->model['individual_cheque_detail']->getRows(array('individual_cheque_id' => $individual_cheque_id), array('employee_name'));

        $pdf = new PDF('P', PDF_UNIT, 'A4', true, 'UTF-8', false);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-Bold.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-BoldItalic.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-ExtraBold.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-ExtraBoldItalic.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-Italic.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-Light.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-LightItalic.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-Regular.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-SemiBold.ttf', 'TrueTypeUnicode', '', 96);
        //TCPDF_FONTS::addTTFfont(K_PATH_FONTS .'/Open_Sans/freesans-SemiBoldItalic.ttf', 'TrueTypeUnicode', '', 96);
        // set document information
        $pdf->SetAuthor('Huzaifa Khambaty');
        $pdf->SetTitle('Bank Letter:'.$voucher['individual_cheque_no'].'@'.date('YmdHis'));
        $pdf->SetSubject('Bank Letter');

        // set margins
        //$pdf->SetMargins(PDF_MARGIN_LEFT, PDF_MARGIN_TOP, PDF_MARGIN_RIGHT);
        $pdf->SetMargins(PDF_MARGIN_LEFT, 10, PDF_MARGIN_RIGHT);
        $pdf->SetHeaderMargin(0);
        $pdf->SetFooterMargin(PDF_MARGIN_FOOTER);

        $pdf->footer = array(
            'individual_cheque_title' => $voucher['individual_cheque_title']
        );
        $pdf->lang = $lang;
        // set auto page breaks
        $pdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM);

        // add a page
        $pdf->AddPage();
        $pdf->SetFont('freesans','',10);
        $pdf->Cell(30,7,stdDate($voucher['individual_cheque_date']));
        $pdf->ln(7);
        $pdf->Cell(60,7,$voucher['bank_name'].',');
        $pdf->ln(7);
        $pdf->Cell(60,7,$voucher['bank_branch']);
        $pdf->ln(14);
        $pdf->Cell(30,7,'Dear Sir,');
        $pdf->ln(7);
        $pdf->Cell(15,7,'');
        $pdf->Cell(150,7,'We are enclosing herewith Cheque No. '.$voucher['cheque_no'].'  Dated: '.stdDate($voucher['cheque_date']));
        $pdf->ln(7);
        $pdf->Cell(15,7,'');
        $pdf->Cell(150,7,'for Rs. '.number_format($voucher['cheque_amount'],2).' drawn on '.$voucher['bank_name'].' being '.$voucher['individual_cheque_title'].'.');
        $pdf->ln(7);
        $pdf->Cell(15,7,'');
        $pdf->Cell(150,7,'kindly credit the following accounts');
        $pdf->ln(14);
        $pdf->SetFont('freesans','B',10);
        $pdf->Cell(30,7,'SNo.','TB','','C');
        $pdf->Cell(95,7,'Account Title','TB','','C');
        $pdf->Cell(25,7,'Account No.','TB','','C');
        $pdf->Cell(30,7,'Rupees','TB','','C');
        $pdf->SetFont('freesans','',10);
        $sr_no=0;
        foreach($details as $detail) {
            $sr_no++;
            $pdf->ln(7);
            $pdf->Cell(30,7,$sr_no,'B','','C');
            $pdf->Cell(95,7,$detail['account_title'],'B','','L');
            $pdf->Cell(25,7,$detail['account_no'],'B','','R');
            $pdf->Cell(30,7,number_format($detail['amount'],2),'B','','R');
        }
        $pdf->ln(14);
        $pdf->Cell(125,7,'','','','R');
        $pdf->Cell(25,7,'Total Rupees:','TB','','R');
        $pdf->Cell(30,7,number_format($voucher['cheque_amount'],2),'TB','','R');
        $pdf->ln(8);
        $pdf->Cell(125,7,'','','','R');
        $pdf->Cell(25,7,'Yours truly','T','','LB');
        $pdf->Cell(30,7,'','T','','R');
        $pdf->ln(20);
        $pdf->Cell(125,7,'','','','R');
        if($this->session->data['parent_company_id'] != 0) {
            $pdf->Cell(55,7,'For DAWAT-E-HADIYAH','','','L');
        } else {
            $pdf->Cell(55,7,$this->session->data['company_name'],'','','R');
        }

        $pdf->Output('Bank Letter:'.$voucher['individual_cheque_no'].'@'.date('YmdHis').'.pdf', 'I');
    }

    public function getAccountsData() {
        $this->model['individual_cheque'] = $this->load->model('payroll/individual_cheque');

        $post = $this->request->post;
        $individual_cheque_id = $this->request->get['individual_cheque_id'];

        $pay_date = $post['pay_date'];

        if($post['source_type']=='payroll') {
            // Payroll have Multiple Credit Entries
            $account_details = $this->model['individual_cheque']->getPostDataPayroll($individual_cheque_id);
            $ctrl_no = 1;
            $data = array();
            $budgets = array();
            foreach($account_details as $detail) {
                if($detail['PartyID']==35) {
                    $detail['Party'] = 'Employee OLD Age Benefits Institution';
                } elseif($detail['PartyID']==105) {
                    $detail['Party'] = 'National bank of Pakistan';
                } elseif($detail['PartyID']==158) {
                    $detail['Party'] = 'Yourself';
                } elseif($detail['PartyID']==28) {
                    $detail['Party'] = 'National Bank of Pakistan';
                }

                if($detail['tran_type']=='D') {
                    if(isset($budgets[$detail['budget_key']])) {
                        $budgets[$detail['budget_key']]['budAmount'] += $detail['Pamount'];
                    } else {
                        $budgets[$detail['budget_key']] = array(
                            'Comp' => $post['Comp'],
                            'Year' => $detail['Year'],
                            'BudKey' => $detail['budget_key'],
                            'Ref' => str_replace('/','-',$detail['SysKey']),
                            'budAmount' => $detail['Pamount'],
                            'Status' => 'F',
                            'PartyID' => $detail['PartyID'],
                            'AHKey' => $detail['AHKey'],
                        );
                        $TranCategory = $detail['TranCategory'];
                    }
                }

                $detail['TranCategory'] = $TranCategory;
                $detail['Comp'] = $post['Comp'];
                //$detail['PayDate'] = date('Ymd',strtotime($detail['PayDate']));
                $detail['PayDate'] = date('Ymd',strtotime($pay_date));
                $detail['SysKey'] = str_replace('/','-',$detail['SysKey']);
                $detail['Ctrl_No'] = $ctrl_no;
                if(isset($data[$detail['AHKey'].'-'.$detail['Ref']])) {
                    $data[$detail['AHKey'].'-'.$detail['Ref']]['Pamount'] += $detail['Pamount'];
                    $data[$detail['AHKey'].'-'.$detail['Ref']]['Bamount'] += $detail['Bamount'];
                } else {
                    $data[$detail['AHKey'].'-'.$detail['Ref']] = $detail;
                }
            }
            //d([$post,$budgets,$data], true);
        } else {
            // Other have Single Credit Entries
            if($post['source_type']=='sp_enayat') {
                $account_details = $this->model['individual_cheque']->getPostDataSpecialEnayat($individual_cheque_id);
            } else {
                $account_details = $this->model['individual_cheque']->getPostDataSpecialAllowance($individual_cheque_id);
            }

            $ctrl_no = 0;
            $budgets = array();
            //d($account_details, true);
            foreach($account_details as $detail) {
                $budgets[] = array(
                    'Comp' => $post['Comp'],
                    'Year' => $detail['Year'],
                    'BudKey' => $detail['budget_key'],
                    'Ref' => str_replace('/','-',$detail['document_identity']),
                    'budAmount' => $detail['cheque_amount'],
                    'Status' => 'F',
                    'PartyID' => $detail['PartyID'],
                    'AHKey' => $detail['debit_ah_key'],
                );
                $ctrl_no++;
                $data[] = array (
                    //'PayDate' => date('Ymd',strtotime($detail['document_date'])),
                    'PayDate' => date('Ymd',strtotime($pay_date)),
                    'SysKey' => str_replace('/','-',$detail['document_identity']),
                    'SysTag' => 'PR',
                    'TranCategory' => $detail['TranCategory'],
                    'Year' => $detail['Year'],
                    //'Comp' => '',
                    'Comp' => $post['Comp'],
                    'PartyID' => $detail['PartyID'],
                    'Party' => str_replace('/','',$detail['Party']),
                    'tran_type' => 'D',
                    'BAHKey' => 0,
                    'AHKey' => $detail['debit_ah_key'],
                    'CCKey' => 0,
                    'BudKey' => 0,
                    'Narr' => $detail['document_title'],
                    'Ref' => '',
                    //'Ref' => $detail['cheque_no'],
                    'taxType' => 'N',
                    'Pamount' => $detail['cheque_amount'],
                    'Bamount' => $detail['cheque_amount'],
                    'Ctrl_No' => $ctrl_no,
                );

                $data[] = array (
                    //'PayDate' => date('Ymd',strtotime($detail['document_date'])),
                    'PayDate' => date('Ymd',strtotime($pay_date)),
                    'SysKey' => str_replace('/','-',$detail['document_identity']),
                    'SysTag' => 'PR',
                    'TranCategory' => $detail['TranCategory'],
                    'Year' => $detail['Year'],
                    //'Comp' => '',
                    'Comp' => $post['Comp'],
                    'PartyID' => $detail['PartyID'],
                    'Party' => str_replace('/','',$detail['Party']),
                    //'tran_type' => 'D',
                    'tran_type' => 'C',
                    'BAHKey' => $detail['credit_ah_key'],
                    'AHKey' => $detail['credit_ah_key'],
                    'CCKey' => 0,
                    'BudKey' => 0,
                    'Narr' => $detail['document_title'],
                    //'Ref' => '',
                    'Ref' => $detail['cheque_no'],
                    'taxType' => 'N',
                    'Pamount' => $detail['cheque_amount'],
                    'Bamount' => $detail['cheque_amount'],
                    'Ctrl_No' => $ctrl_no,
                );
            }
        }


        $json = [
            'success' => true,
            'budgets' => $budgets,
            'accounts' => $data
        ];

        echo json_encode($json, true);
        exit;
    }

    public function post() {
        parent::getForm();
        $individual_cheque_id = $this->request->get['individual_cheque_id'];

        if (($this->request->server['REQUEST_METHOD'] == 'POST')) {
            $this->db->beginTransaction();
            $id = $this->postData($this->request->post);
            $this->db->commit();

            //$this->session->data['success'] = $this->language->get('success_insert');
            $this->redirect($this->url->link($this->getAlias(), 'token=' . $this->session->data['token'], 'SSL'));
        }

        $this->data['lang'] = $this->load->language($this->getAlias());
        $this->model['individual_cheque'] = $this->load->model('payroll/individual_cheque');
        $this->model['company_account'] = $this->load->model('setup/company_account');
        $this->model['individual_cheque_detail'] = $this->load->model('payroll/individual_cheque_detail');
        $this->model['company'] = $this->load->model('setup/company');

        $result = $this->model['individual_cheque']->getRow(['individual_cheque_id' => $individual_cheque_id]);
        $company_account = $this->model['company_account']->getRow(['company_account_id' => $result['company_account_id']]);
        $company = $this->model['company']->getRow(['company_id' => $company_account['company_id']]);

        //d([$company, $company_account], true);
        foreach($result as $field => $value) {
            if($field=='document_date') {
                $this->data[$field] = stdDate($value);
            } else {
                $this->data[$field] = $value;
            }
        }
        foreach($company_account as $field => $value) {
            if($field=='document_date') {
                $this->data[$field] = stdDate($value);
            } else {
                $this->data[$field] = $value;
            }
        }

        $this->data['pay_from_company_id'] = $company['ref_id'];
        $this->data['companies'] = $this->model['company']->getRows(array(), array('company_name'));
        $this->data['cheque_details'] = $this->model['individual_cheque_detail']->getRows(['individual_cheque_id' => $individual_cheque_id]);

        $this->data['href_get_accounts_data'] = $this->url->link($this->getAlias() . '/getAccountsData', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $this->request->get[$this->getPrimaryKey()], 'SSL');
        $this->data['action_save'] = $this->url->link($this->getAlias() . '/post', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $this->request->get[$this->getPrimaryKey()], 'SSL');
        $this->template = $this->getAlias() . '_post.tpl';
        $this->response->setOutput($this->render());
    }

    private function postData($data) {
        $this->model['individual_cheque'] = $this->load->model('payroll/individual_cheque');
        try {
            $file_name = DIR_ROOT ."ICPosting_".date('YmdHi').".txt";
            $file = fopen($file_name,"a+");
            fwrite($file,'Request:'.date('YmdHis')."\n");
            fwrite($file,json_encode(array('ledgers'=>$data['account_details'], 'budgets'=>$data['budgets'])));
            fclose($file);

            $client = new SoapClient(ACC_SERV_URL);
            $response = $client->Payroll(array('Payrolldata' => json_encode($data['account_details']), 'Payrollbuddata' => json_encode($data['budgets'])));
            //d($response, true);
            $file = fopen($file_name,"a+");
            fwrite($file,'Response:'.date('YmdHis'));
            fwrite($file,json_encode($response)."\n");
            fclose($file);
            if($response->PayrollResult->string[0]=='Success') {
                $data = array(
                    'is_post' => 1,
                    'post_at' => date('Y-m-d H:i:s'),
                    'post_by_id' => $this->session->data['user_id'],
                    'bno' => $response->PayrollResult->string[1]
                );
                $this->model['individual_cheque']->edit($this->getAlias(),$this->request->get[$this->getPrimaryKey()],$data);
                $this->session->data['success'] = 'Data Posted Successfully on BNO: '. $response->PayrollResult->string[1];
            } else {
                $this->session->data['error_warning'] = $response->PayrollResult->string[1];
            }
        } catch (Exception $e) {
            d($e, true);
            $this->session->data['error_warning'] = $e->getMessage();
        }
    }

    public function printDocument() {
        ini_set('max_execution_time',0);
        ini_set('memory_limit',-1);

        $lang = $this->load->language($this->getAlias());
        $individual_cheque_id = $this->request->get['individual_cheque_id'];
        $post = $this->request->post;
        $session = $this->session->data;

//        $this->model['company'] = $this->load->model('setup/company');
        $this->model['individual_cheque'] = $this->load->model('payroll/individual_cheque');
        $this->model['individual_cheque_detail'] = $this->load->model('payroll/individual_cheque_detail');


//        $company = $this->model['company']->getRow(array('company_id' => $this->session->data['company_id']));
        $individual_cheque = $this->model['individual_cheque']->getRow(array('individual_cheque_id' => $individual_cheque_id));
        $individual_cheque_detail = $this->model['individual_cheque_detail']->getRows(array('individual_cheque_id' => $individual_cheque_id));
        //d(array($individual_cheque, $individual_cheque_detail), true);

        $this->model['allowance_type'] = $this->load->model('setup/special_allowance_type');

        $allowance_types = $this->model['allowance_type']->getRows(array(), array('special_allowance_type'));

        $source_types = array(
            'payroll' => 'Payroll',
            'sp_enayat' => 'Special Enayat',
        );
        foreach($allowance_types as $allowance_type) {
            $source_types[$allowance_type['special_allowance_type_id']] = $allowance_type['special_allowance_type'];
        }

      //  d(array($individual_cheque, $source_types[2]), true);

        $pdf = new PDF('P', PDF_UNIT, 'A4', true, 'UTF-8', false);

        // set document information
        $pdf->SetCreator(PDF_CREATOR);
        $pdf->SetAuthor('Fahad Siddiqui');
        $pdf->SetTitle('Individual Cheque');
        $pdf->SetSubject('Individual Cheque');

        //Set Header
        $pdf->data = array(
            'company_name' => $this->session->data['company_name'] ,
            'report_name' => 'Individual Cheque',
//            'company_logo' => $session['company_image'],
            'status' => 'Print',
        );


        // set margins
        //$pdf->SetMargins(PDF_MARGIN_LEFT, PDF_MARGIN_TOP, PDF_MARGIN_RIGHT);
        $pdf->SetMargins(10, 5, 10);
        $pdf->SetHeaderMargin(2);
        $pdf->SetFooterMargin(2);

        // set auto page breaks
        //$pdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM);

        // set font
        $pdf->SetFont('times', 'B', 10);

        // add a page
        $pdf->AddPage();

        $pdf->SetTextColor(0,0,0);
        $pdf->SetFont('times', 'B', 15);
        $pdf->Ln(2);
        $pdf->Cell(207,6, 'DAWAT-E-HADIYAH', 0, false, 'C', 0, '', 0, false, 'M', 'M');
        $pdf->Ln(7);
        $pdf->SetFont('times', 'B', 14);
        $pdf->Cell(207,6, $pdf->data['report_name'], 0, false, 'C', 0, '', 0, false, 'M', 'M');


        $pdf->SetFont('times', '', 10);
        $pdf->Ln(10);
        $pdf->Cell(26,6, 'Document Title : ', 0, false, 'L', 0, '', 0, false, 'M', 'M');
        $pdf->Cell(169,6, $individual_cheque['document_title'], 'B', false, 'L', 0, '', 0, false, 'M', 'M');

        $pdf->Ln(9);
        $pdf->Cell(25,6, 'Document No. : ', 0, false, 'L', 0, '', 0, false, 'M', 'M');
        $pdf->Cell(30,6, $individual_cheque['document_identity'], 'B', false, 'L', 0, '', 0, false, 'M', 'M');
        $pdf->Cell(83,6, '', 0, false, 'L', 0, '', 0, false, 'M', 'M');
        $pdf->Cell(26,6, 'Document Date : ', 0, false, 'L', 0, '', 0, false, 'M', 'M');
        $pdf->Cell(30,6, stdDate($individual_cheque['document_date']), 'B', false, 'L', 0, '', 0, false, 'M', 'M');
        $pdf->Ln(9);
        $pdf->Cell(25,6, 'Source Type  : ', 0, false, 'L', 0, '', 0, false, 'M', 'M');
        $pdf->Cell(30,6, $source_types[$individual_cheque['source_type']], 'B', false, 'L', 0, '', 0, false, 'M', 'M');
        $pdf->Cell(83,6, '', 0, false, 'L', 0, '', 0, false, 'M', 'M');
        $pdf->Cell(26,6, 'Source Date : ', 0, false, 'L', 0, '', 0, false, 'M', 'M');
        $pdf->Cell(30,6, stdDate($individual_cheque['source_date']), 'B', false, 'L', 0, '', 0, false, 'M', 'M');
//        $pdf->Cell(10,6, '', 0, false, 'L', 0, '', 0, false, 'M', 'M');
//        $pdf->Cell(25,6, 'Enayat Event : ', 0, false, 'L', 0, '', 0, false, 'M', 'M');
//        $pdf->Cell(26,6, $individual_cheque['enayat_event'], 'B', false, 'L', 0, '', 0, false, 'M', 'M');

        $pdf->Ln(10);
        $pdf->SetFont('times', 'B', 9);
        $pdf->Cell(10, 8, 'No.', 1, false, 'C', 0, '', 0, false, 'M', 'M');
        $pdf->Cell(110, 8, 'Employee Name', 1, false, 'C', 0, '', 0, false, 'M', 'M');
        $pdf->Cell(25, 8, 'Cheque Date', 1, false, 'C', 0, '', 0, false, 'M', 'M');
        $pdf->Cell(25, 8, 'Cheque No	', 1, false, 'C', 0, '', 0, false, 'M', 'M');
        $pdf->Cell(25, 8, 'Cheque Amount', 1, false, 'C', 0, '', 0, false, 'M', 'M');
        $sr = 0;
        $pdf->Ln(7);
        $Amount = 0;
        $pdf->SetFont('times', '', 9);
        foreach($individual_cheque_detail as $detail) {
            if($detail['cheque_amount'] > 0)
            {
                $sr++;
                $pdf->Cell(10, 5, $sr, 'LR', false, 'C', 0, '', 0, false, 'M', 'M');
                $pdf->Cell(110, 5, empty($detail['employee_name'])?$detail['bank_title']:$detail['employee_name'], 'LR',false, 'L', 0, '', 0, false, 'M', 'M');
                $pdf->Cell(25, 5, stdDate($detail['cheque_date']), 'LR',false, 'L', 0, '', 0, false, 'M', 'M');
                $pdf->Cell(25, 5, $detail['cheque_no'], 'LR',false, 'L', 0, '', 0, false, 'M', 'M');
                $pdf->Cell(25, 5, number_format($detail['cheque_amount'],2), 'LR', false, 'R', 0, '', 0, false, 'M', 'M');
                $pdf->Ln(5);
            }
        }

        $x = $pdf->GetX();
        $y = $pdf->GetY();

        for ($i = $y; $i <= 260; $i++) {

            $pdf->Ln(1);
            $pdf->Cell(10, 7, '', 'L',  false, 'C', 0, '', 0, false, 'M', 'M');
            $pdf->Cell(110, 7,'', 'L', false, 'L', 0, '', 0, false, 'M', 'M');
            $pdf->Cell(25, 7,'', 'L', false, 'L', 0, '', 0, false, 'M', 'M');
            $pdf->Cell(25, 7,'', 'L', false, 'L', 0, '', 0, false, 'M', 'M');
            $pdf->Cell(25, 7, '', 'LR', false, 'L', 0, '', 0, false, 'M', 'M');
            $y =$i;
        }
        $pdf->Ln(-1);
        $pdf->Ln(1);
        $pdf->Cell(195, 7, '', 'B', false, 'C', 0, '', 0, false, 'M', 'M');
        $pdf->setXY($x,$y);

        $pdf->SetFont('times', 'B', 8);
        $pdf->Ln(8);
        $pdf->Cell(170, 7, '', 0, false, 'R', 0, '', 0, false, 'M', 'M');
        $pdf->Cell(25, 7, number_format($individual_cheque['total_amount'],2), 1, false, 'R', 0, '', 0, false, 'M', 'M');

        $data= $individual_cheque;
        //Close and output PDF document
        $pdf->Output('Individual Cheque :'.date('YmdHis').'.pdf', 'I');
    }

    public function viewSMS() {
        parent::getForm();
        $individual_cheque_id = $this->request->get['individual_cheque_id'];
        $this->model['employee_registration'] = $this->load->model('employee/employee_registration');
        $this->model['individual_cheque'] = $this->load->model('payroll/individual_cheque');
        $this->data['lang'] = $this->load->language($this->getAlias());
        $this->data['employees'] = $this->model['individual_cheque']->getEmployeeData($individual_cheque_id);
        $this->data['href_send_sms'] = $this->url->link($this->getAlias() . '/sendSMS', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $this->request->get[$this->getPrimaryKey()], 'SSL');

        //d($this->data['employees'], true);
        $bl = $this->model['individual_cheque']->getRow(['individual_cheque_id' => $individual_cheque_id]);
        if($bl['source_type']=='payroll') {
            $this->data['default_message'] = "Your ".$bl['document_title']." is ready, please collect.";
        } elseif($bl['source_type']=='sp_enayat') {
            $this->data['default_message'] = "Your Enayat Cheque is ready, please collect.";
        } else {
            $this->data['default_message'] = "Your Allowance Cheque is ready, please collect";
        }

        $this->model['core_setting'] = $this->load->model('setup/core_setting');
        $config = $this->model['core_setting']->getConfig(['company_id' => $this->session->data['company_id']]);
        $user_name = $config['sms_user_name'];
        $password = $config['sms_password'];
        $from_mask = $config['sms_from_mask'];

        if($user_name=='' || $password=='' || $from_mask=='') {
            $this->data['warning'] = "Please verify you have valid SMS Setting configured.";
        }
        $this->template = $this->getAlias() . '_sms.tpl';
        $this->response->setOutput($this->render());
    }

    public function sendSMS() {
        $post = $this->request->post;
        $employee = $post['employee'];
        $message = $post['message'];

        $this->model['core_setting'] = $this->load->model('setup/core_setting');
        $config = $this->model['core_setting']->getConfig(['company_id' => $this->session->data['company_id']]);
        $user_name = $config['sms_user_name'];
        $password = $config['sms_password'];
        $from_mask = $config['sms_from_mask'];

        //d($config, true);
        $mobile_no = str_replace("+92","0",str_replace(" ","",$employee['mobile_no']));
        //$mobile_no = '03332221175';
        $url = 'https://connect.jazzcmt.com/sendsms_url.html?Username='.$user_name.'&Password='.$password.'&From='.$from_mask.'&To='.$mobile_no.'&Message=' . urlencode($message);
        $response = file_get_contents($url);
        $json['success'] = true;
        $json['mobile_no'] = $mobile_no;
        $json['response'] = $response;
        echo json_encode($json);
        exit;
    }

    public function viewEmail() {
        parent::getForm();
        $individual_cheque_id = $this->request->get['individual_cheque_id'];
        $this->model['employee_registration'] = $this->load->model('employee/employee_registration');
        $this->model['individual_cheque'] = $this->load->model('payroll/individual_cheque');
        $this->data['lang'] = $this->load->language($this->getAlias());
        $this->data['employees'] = $this->model['individual_cheque']->getEmployeeData($individual_cheque_id);
        $this->data['href_send_email'] = $this->url->link($this->getAlias() . '/sendEmail', 'token=' . $this->session->data['token'] . '&' . $this->getPrimaryKey() . '=' . $this->request->get[$this->getPrimaryKey()], 'SSL');

        $this->model['core_setting'] = $this->load->model('setup/core_setting');
        $config = $this->model['core_setting']->getConfig(['company_id' => $this->session->data['company_id']]);
        $user_name = $config['sms_user_name'];
        $password = $config['sms_password'];
        $from_mask = $config['sms_from_mask'];

        if($user_name=='' || $password=='' || $from_mask=='') {
            $this->data['warning'] = "Please verify you have valid SMTP and SMS Setting configured.";
        }

        $this->template = $this->getAlias() . '_email.tpl';
        $this->response->setOutput($this->render());
    }

    public function sendEmail() {
        $post = $this->request->post;
        if($post['source_type']=='payroll') {
            $this->model['bank'] = $this->load->model('setup/bank');
            $this->model['payroll'] = $this->load->model('payroll/payroll');
            $this->model['payroll_allowance'] = $this->load->model('payroll/payroll_allowance');
            $this->model['payroll_deduction'] = $this->load->model('payroll/payroll_deduction');
            //d($post, true);
            $post['payroll_id'] = $post['source_table_id'];
            $arrBanks = $this->model['bank']->getArrays('bank_id','bank_name',array());

            $payroll = $this->model['payroll']->getRow(['payroll_id' => $post['payroll_id']]);
            $allowances = $this->model['payroll_allowance']->getRows(array('payroll_id' => $payroll['payroll_id']));
            $deductions = $this->model['payroll_deduction']->getRows(array('payroll_id' => $payroll['payroll_id']));
            $html_body = '';
            $html_body .= 'Employee Code: '. $payroll['registration_code'].'<br />';
            $html_body .= 'Employee Name: '. $payroll['employee_name'].'<br />';
            $html_body .= 'Department Name: '. $payroll['department_name'].'<br />';
            $html_body .= 'Email: '. $payroll['email'].'<br />';
            $html_body .= 'Basic Salary: '. number_format($payroll['basic_salary'],2).'<br />';
            $gross_amount = $payroll['basic_salary'];
            if($payroll['house_rent_allowance'] != 0) {
                $html_body .= 'House Rent: '. number_format($payroll['house_rent_allowance'],2).'<br />';
                //$gross_amount += $payroll['house_rent_allowance'];
            }
            foreach($allowances as $allowance) {
                if($allowance['allowance_amount'] != 0 && $allowance['grossable']) {
                    $html_body .= $allowance['allowance_name'] . ': '. number_format($allowance['allowance_amount'],2).'<br />';
                    $gross_amount += $payroll['allowance_amount'];
                }
            }
            if($payroll['arrears'] != 0) {
                $html_body .= 'Arrears: '. number_format($payroll['arrears'],2).'<br />';
                $gross_amount += $payroll['arrears'];
            }
            if($payroll['manual_addition'] != 0) {
                $html_body .= 'Other Allowance: '. number_format($payroll['manual_addition'],2).'<br />';
                $gross_amount += $payroll['manual_addition'];
            }
            if($payroll['overtime'] != 0) {
                $html_body .= 'Overtime: '. number_format($payroll['overtime'],2).'<br />';
                $gross_amount += $payroll['overtime'];
            }
            $html_body .= '<b>Gross Amount:</b> '. number_format($gross_amount,2).'<br />';
            foreach($deductions as $deduction) {
                if($deduction['deduction_amount'] != 0) {
                    $html_body .= $deduction['deduction_name'] . ': '. number_format($deduction['deduction_amount'],2).'<br />';
                }
            }
            if($payroll['manual_deduction'] != 0) {
                $html_body .= 'Other Deduction: '. number_format($payroll['manual_deduction'],2).'<br />';
            }
            if($payroll['attendance_deduction'] != 0) {
                $html_body .= 'Absent Deduction: '. number_format($payroll['attendance_deduction'],2).'<br />';
            }
            if($payroll['tax_deduction'] != 0) {
                $html_body .= 'Tax: '. number_format($payroll['tax_deduction'],2).'<br />';
            }
            $html_body .= '<b>Net Salary: </b>'. number_format($payroll['payable_salary'],2).'<br />';
            $html_body .= 'Cheque is ready, please collect.';
        } else {
            $html_body = '';
            $html_body .= 'Employee Code: '. $post['registration_code'].'<br />';
            $html_body .= 'Employee Name: '. $post['employee_name'].'<br />';
            $html_body .= 'Department Name: '. $post['department_name'].'<br />';
            $html_body .= 'Email: '. $post['email'].'<br />';
            $html_body .= 'Amount: '. number_format($post['amount'],2).'<br />';
            $html_body .= 'Cheque is ready, please collect.';
        }


        $this->model['core_setting'] = $this->load->model('setup/core_setting');
        $config = $this->model['core_setting']->getArrays('field','value',array('company_id' => $this->session->data['company_id'], 'module'=>'general'));
        //d($config, true);
        try {
            $mail = new PHPMailer(true);                              // Passing `true` enables exceptions
            $mail->SMTPOptions = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );
            //Server settings
            $mail->SMTPDebug = 0;                                 // Enable verbose debug output
            $mail->isSMTP();                                      // Set mailer to use SMTP
            $mail->Host = $config['config_mail_smtp_hostname'];
            $mail->SMTPAuth = true;
            $mail->Username = $config['config_mail_smtp_username']; //paste one generated by Mailtrap
            $mail->Password = $config['config_mail_smtp_password']; //paste one generated by Mailtrap
            //$mail->SMTPSecure = 'tls';
            $mail->Port = $config['config_mail_smtp_port'];
            $mail->setFrom($config['config_mail_smtp_username'], $post['config_mail_smtp_displayname']);
            $mail->addAddress($post['email']);
            //$mail->addAddress('huzaifa.khambaty@gmail.com');
            $mail->isHTML(true);
            // Set email format to HTML
            $mail->Subject = $post['document_title'];
            $mail->Body    = $html_body;
            if(!$mail->Send()) {
                $json = [
                    'success' => false,
                    'error' => $mail->ErrorInfo,
                    'id' => $post['id'],
                    'next_id' => $post['id']+1,
                ];
            } else {
                $json = [
                    'success' => true,
                    'id' => $post['id'],
                    'next_id' => $post['id']+1,
                ];
            }
        } catch (Exception $e) {
            $json = [
                'success' => false,
                'error' => $e->getCode().'|'.$e->getMessage(),
                'id' => $post['id'],
                'next_id' => $post['id']+1,
            ];
        }


        echo json_encode($json);
    }

}

class PDF extends TCPDF {
    public $lang;
    public $header;
    public $footer;
    //Page header
    public function Header() {

    }

    // Page footer
    public function Footer() {
        $data = $this->footer;
        // Position at 15 mm from bottom
        $this->SetY(-15);
        // Set font
        $this->SetFont('freesans', 'I', 8);
        // Page number
        $this->Cell(60, 10, $this->lang['ref'].': '.$data['special_enayat_title'], 0, false, 'L', 0, '', 0, false, 'T', 'M');
        $this->Cell(60, 10, 'Page '.$this->getAliasNumPage().'/'.$this->getAliasNbPages(), 0, false, 'C', 0, '', 0, false, 'T', 'M');
        $this->Cell(60, 10, 'Date: '.date('d-m-Y H:i'), 0, false, 'R', 0, '', 0, false, 'T', 'M');
    }
}
?>