<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\Employee;
use App\Models\EmployeeLeave;
use App\Models\Attendance;
use Illuminate\Http\Request;
use App\Models\EmployeeRegistration;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use PHPMailer\PHPMailer\Exception;

class LeaveController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:api', ['except' => ['postLogin', 'refresh', 'logout']]);
    }

    public function createLeaveRequest(Request $request)
    {
        $this->validate($request, [
            'company_id' => 'required|string',
            'employee_id' => 'required|string',
            'employee_registration_id' => 'required|string',
            'leave_from' => 'required|date_format:d-m-Y',
            'leave_to' => 'required|date_format:d-m-Y',
            'reason' => 'required|string',
            'description' => 'required|string',
            'status' => 'in:Pending,Requested',
        ]);

        $company_id = $request->company_id;
        $employee_id = $request->employee_id;
        $employee_registration_id = $request->employee_registration_id;
        $leave_from = $request->leave_from;
        $leave_to = $request->leave_to;
        $reason = $request->reason;
        $description = $request->description;
        $status = $request->status?$request->status:'Pending';

        $objRegistration = EmployeeRegistration::where('company_id', $company_id)
            ->where('employee_id','=',$employee_id)
            ->where('employee_registration_id','=',$employee_registration_id)
            ->where('status','=','Active')
            ->first();

        if($objRegistration) {
            $objCompany = Company::find($company_id);
            date_default_timezone_set($objCompany->time_zone);

            $employee_leave_id = Str::uuid();

            $dataLeave = [
                'company_id' => $company_id,
                'employee_registration_id' => $employee_registration_id,
                'employee_leave_id' => $employee_leave_id,
                'leave_from' => MySqlDate($leave_from),
                'leave_to' => MySqlDate($leave_to),
                'reason' => $reason,
                'description' => $description,
                'request_date' => date('Y-m-d'),
                'status' => $status,
                'created_at' => date('Y-m-d H:i:s'),
                'created_by_id' => NULL,
            ];

            DB::beginTransaction();
            EmployeeLeave::insert($dataLeave);
            DB::commit();

            $response = [
                'data' => [
                    'employee_leave_id' => $employee_leave_id
                ]
            ];
            return response()->json($response, 200);
        } else {
            return response()->json('Invalid request', 400);
        }
    }

    public function updateLeaveRequest(Request $request, $id) {
        try {
            $this->validate($request, [
                'leave_from' => 'required|date_format:d-m-Y',
                'leave_to' => 'required|date_format:d-m-Y',
                'reason' => 'required|string',
                'description' => 'required|string',
                'status' => 'in:Pending,Requested',
            ]);

//            $company_id = $request->company_id;
//            $employee_id = $request->employee_id;
//            $employee_registration_id = $request->employee_registration_id;
//            $leave_from = $request->leave_from;
//            $leave_to = $request->leave_to;
//            $reason = $request->reason;
//            $description = $request->description;
//            $status = $request->status?$request->status:'Pending';

            $objLeave = EmployeeLeave::find($id);
            if($objLeave) {
                $objCompany = Company::find($objLeave->company_id);
                date_default_timezone_set($objCompany->time_zone);
                if(isset($request->leave_from) && $request->leave_from != '') {
                    $objLeave->leave_from = MySqlDate($request->leave_from);
                }
                if(isset($request->leave_to) && $request->leave_to != '') {
                    $objLeave->leave_to = MySqlDate($request->leave_to);
                }
                if(isset($request->reason) && $request->reason != '') {
                    $objLeave->reason = $request->reason;
                }
                if(isset($request->description) && $request->description != '') {
                    $objLeave->description = $request->description;
                }
                if(isset($request->status) && $request->status != '') {
                    $objLeave->status = $request->status;
                }

                DB::beginTransaction();
                $objLeave->save();
                DB::commit();

                $data = [
                    'id' => $objLeave->employee_leave_id,
                    'leave_from' => stdDate($objLeave->leave_from),
                    'leave_to' => stdDate($objLeave->leave_to),
                    'reason' => $objLeave->reason,
                    'description' => $objLeave->description,
                    'request_date' => stdDate($objLeave->request_date),
                    'status' => $objLeave->status,
                    'created_at' => stdDateTime($objLeave->created_at),
                ];
                return response()->json($data, 200);
            } else {
                return response()->json('Invalid request', 400);
            }

        } catch (Exception $e) {
            DB::rollBack();
            return $this->jsonResponse([],$e->getCode(),$e->getMessage());
        }
    }

    public function getLeaveRequest(Request $request, $employee_leave_id) {
        $this->validate($request, [
            'company_id' => 'required|string',
            'employee_id' => 'required|string',
            'employee_registration_id' => 'required|string',
        ]);

        $company_id = $request->company_id;
        $employee_id = $request->employee_id;
        $employee_registration_id = $request->employee_registration_id;

        $arrApprovedTypes = [
            'LD' => 'Leave with Deduction',
            'LP' => 'Leave with Pay',
            'UA' => 'Leave Unavailed',
            'WV' => 'Leave Waived',
            'HD' => 'Half Day',
            'WF' => 'Work from Home',
        ];
        $objLeave = EmployeeLeave::where('employee_leave_id','=',$employee_leave_id)
            ->where('company_id','=',$company_id)
            ->where('employee_registration_id','=',$employee_registration_id)
            ->first();
        if($objLeave && $objLeave->company_id == $company_id) {
            $objRegistration = EmployeeRegistration::find($objLeave->employee_registration_id);
            if($objRegistration) {
                $objEmployee = Employee::find($employee_id);
                if($objEmployee) {
                    $data = [
                        'id' => $objLeave->employee_leave_id,
                        'leave_from' => stdDate($objLeave->leave_from),
                        'leave_to' => stdDate($objLeave->leave_to),
                        'reason' => $objLeave->reason,
                        'description' => $objLeave->description,
                        'status' => $objLeave->status,
                        'approved_type' => ($objLeave->approved_type?$arrApprovedTypes[$objLeave->approved_type]:NULL),
                        'approved_from' => ($objLeave->approved_from?stdDate($objLeave->approved_from):null),
                        'approved_to' => ($objLeave->approved_to?stdDate($objLeave->approved_to):null),
                        'remarks' => $objLeave->remarks,
                        'created_at' => ($objLeave->created_at?stdDateTime($objLeave->created_at):null),
                    ];

                    return response()->json(['data'=>$data], 200);
                } else {
                    return response()->json('Invalid request', 400);
                }
            } else {
                return response()->json('Invalid request', 400);
            }
        } else {
            return response()->json('Invalid request', 400);
        }
    }

    public function getLeaveRequests(Request $request) {
        $this->validate($request, [
            'company_id' => 'required|string',
            'employee_id' => 'required|string',
            'employee_registration_id' => 'required|string',
            'from_date' => 'date_format:d-m-Y',
            'to_date' => 'date_format:d-m-Y',
            'order_by' => 'string',
            'sort_by' => 'string',
            'page' => 'integer',
            'limit' => 'integer',
        ]);

        $company_id = $request->company_id;
        $employee_id = $request->employee_id;
        $employee_registration_id = $request->employee_registration_id;
        $date_from = ($request->from_date?MySqlDate($request->from_date):NULL);
        $date_to = ($request->to_date?MySqlDate($request->to_date):NULL);
        $order_by = ($request->order_by?$request->order_by:'created_at');
        $sort_by = ($request->sort_by?$request->sort_by:'DESC');
        $page = ($request->page?$request->page:1);
        $limit = ($request->limit?$request->limit:10);
        $offset = ($page-1)*$limit;

        $arrApprovedTypes = [
            'LD' => 'Leave with Deduction',
            'LP' => 'Leave with Pay',
            'UA' => 'Leave Unavailed',
            'WV' => 'Leave Waived',
            'HD' => 'Half Day',
            'WF' => 'Work from Home',
        ];
        $objLeave = EmployeeLeave::where('company_id','=',$company_id)
            ->where('employee_registration_id','=',$employee_registration_id);
        if($date_from) {
            $objLeave = $objLeave->where('created_at','>=',$date_from.' 00:00:00');
        }
        if($date_to) {
            $objLeave = $objLeave->where('created_at','<=',$date_to.' 23:59:59');
        }
        $objLeave = $objLeave->orderBy($order_by,$sort_by)
            ->skip($offset)
            ->take($limit);

        $leaves = $objLeave->get();

        if($leaves) {
            $data = [];
            foreach($leaves as $leave) {
                $data[] = [
                    'id' => $leave->employee_leave_id,
                    'leave_from' => stdDate($leave->leave_from), //Carbon::createFromFormat('Y-m-d', $leave->leave_from)->format('d-m-Y'),
                    'leave_to' => stdDate($leave->leave_to), //Carbon::createFromFormat('Y-m-d', $leave->leave_to)->format('d-m-Y'),
                    'reason' => $leave->reason,
                    'description' => $leave->description,
                    'status' => $leave->status,
                    'approved_type' => ($leave->approved_type?$arrApprovedTypes[$leave->approved_type]:NULL),
                    'approved_from' => ($leave->approved_from?stdDate($leave->approved_from):null),
                    'approved_to' => ($leave->approved_to?stdDate($leave->approved_to):null),
                    'remarks' => $leave->remarks,
                    'created_at' => ($leave->created_at?stdDateTime($leave->created_at):null),
                ];
            }
            return response()->json(['data'=>$data], 200);
        } else {
            return response()->json('Invalid request', 400);
        }
    }
	
	    public function availLeave(Request $request)
    {
        $employee_id = $request->employee_id;
        $employeesRegistration = EmployeeRegistration::where('employee_id', $employee_id)->get();
        if (count($employeesRegistration) == 0) {
            return response()->json([
                'message' => 'Employee not found',
                'status' => 'Faild',
                'status_code' => 404
            ], 404);
        }
        
        $availLeaves = Attendance::where('employee_registration_id', $employeesRegistration[0]->employee_registration_id)
             ->where('company_id','=', $request->company_id)
            ->whereBetween('attendance_date', [date('Y-01-01'), date('Y-12-t')])
            ->whereIn('status', ['LP','A','HD'])
            ->orderBy('status')->get();

        $leaveCount = 0;
        $halfDay = 0;
        foreach ($availLeaves as $row) {
            $leaveCount+= ($row['status']!="HD") ? 1 : 0 ;
            $halfDay+= ($row['status']=="HD") ? 1 : 0 ;
            if($halfDay==2){
                $leaveCount ++;
                $halfDay = 0;
            }
        }

        $lateMonthWise = Attendance::where('employee_registration_id', $employeesRegistration[0]->employee_registration_id)
            ->where('company_id','=', $request->company_id)
            ->whereBetween('attendance_date', [date('Y-01-01'), date('Y-12-t')])
            ->where('status','LT')
            ->orderBy('attendance_date')->get();

        $lateFee = [];
        $total_late = 0;
        $monthWiseLate = [];
        foreach ($lateMonthWise as $row) {
  
            if(isset($lateFee[date('M',strtotime($row['attendance_date']))])){
                 $lateFee[date('M',strtotime($row['attendance_date']))]+= 1;
                 $monthWiseLate[date('M',strtotime($row['attendance_date']))] = ['name'=>date('M',strtotime($row['attendance_date'])),'value'=>$lateFee[date('M',strtotime($row['attendance_date']))]];
             }else{
                $lateFee[date('M',strtotime($row['attendance_date']))] = 1;
                $monthWiseLate[date('M',strtotime($row['attendance_date']))] = ['name'=>date('M',strtotime($row['attendance_date'])),'value'=>1];
             }

             if($lateFee[date('M',strtotime($row['attendance_date']))]%3==0){
                $leaveCount++;
                $total_late++;
             }
        }
        
        return response()->json([
                'data' =>   ['total_month_wise_late'=>$monthWiseLate, 'total_late_leaves' => $total_late,'total_leaves'=>15,'avail_leaves'=>$leaveCount,'remaining_leaves'=>15-$leaveCount],
                'message' =>   "Employee Total Leaves ",
                'status' => 'Success',
                'status_code' => 200
            ], 200);

    }
}