<?php

namespace App\Http\Controllers;

use App\Models\Project;
use App\Models\ProjectSection;
use App\Models\Task;
use App\Models\TaskComment;
use App\Models\Team;
use App\Models\TeamMember;
use Carbon\Carbon;
use App\Models\Employee;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use PHPMailer\PHPMailer\Exception;
use Symfony\Component\Console\Helper\Helper;

class TaskManagementController extends Controller
{

    public function __construct()
    {
        $this->middleware('auth:api', ['except' => ['postLogin', 'refresh', 'logout']]);
    }

    public function getTeams(Request $request) {
        $this->validate($request, [
            'company_id' => 'required|string',
            'employee_id' => 'required|string',
        ]);

        $company_id = $request->company_id;
        $employee_id = $request->employee_id;

        $data = [];
        $teams = TeamMember::where('company_id','=',$company_id)->where('employee_id','=',$employee_id)->get();
        foreach($teams as $row) {
            $team = Team::find($row->team_id);
            $data[] = [
                'team_id' => $row->team_id,
                'name' => $team->name,
                'description' => $team->description,
                'html_description' => $team->html_description,
            ];
        }

        return $this->jsonResponse($data,200);
    }

    public function getProjectsThroughTeam(Request $request) {
        $this->validate($request, [
            'company_id' => 'required|string',
            'employee_id' => 'required|string',
            'team_id' => 'required|string',
        ]);

        $company_id = $request->company_id;
        $employee_id = $request->employee_id;
        $team_id = $request->team_id;

        $data = [];
        $row = TeamMember::where('company_id','=',$company_id)->where('employee_id','=',$employee_id)->where('team_id','=',$team_id)->first();
        $team = Team::find($row->team_id);
        $projects = Project::where('company_id','=',$company_id)->where('team_id','=',$row->team_id)->get();
        $arrProjects = [];
        foreach($projects as $project) {
            $arrProjects[] = [
                'project_id' => $project->project_id,
                'name' => $project->name,
                'description' => $project->description,
                'html_description' => $project->html_description,
                'color' => $project->color,
                'archived' => $project->archived,
            ];
        }
        $data = [
            'team_id' => $row->team_id,
            'name' => $team->name,
            'description' => $team->description,
            'html_description' => $team->html_description,
            'projects' => $arrProjects,
        ];
        return $this->jsonResponse($data,200);
    }

    public function getProjectsWithTeam(Request $request) {
        $this->validate($request, [
            'company_id' => 'required|string',
            'employee_id' => 'required|string',
        ]);

        $company_id = $request->company_id;
        $employee_id = $request->employee_id;

        $data = [];
        $teams = TeamMember::where('company_id','=',$company_id)->where('employee_id','=',$employee_id)->get();
        foreach($teams as $row) {
            $team = Team::find($row->team_id);
            $projects = Project::where('company_id','=',$company_id)->where('team_id','=',$row->team_id)->get();
            $arrProjects = [];
            foreach($projects as $project) {
                $sections = ProjectSection::where('company_id','=',$company_id)->where('project_id','=',$project->project_id)->get();
                $arrSections = [];
                foreach($sections as $section) {
                    $arrSections[] = [
                        'project_section_id' => $section->project_section_id,
                        'section_name' => $section->section_name,
                    ];
                }
                $arrProjects[] = [
                    'project_id' => $project->project_id,
                    'name' => $project->name,
                    'description' => $project->description,
                    'html_description' => $project->html_description,
                    'color' => $project->color,
                    'archived' => $project->archived,
                    'sections' => $arrSections
                ];
            }
            $data[] = [
                'team_id' => $row->team_id,
                'name' => $team->name,
                'description' => $team->description,
                'html_description' => $team->html_description,
                'projects' => $arrProjects,
            ];
        }
        return $this->jsonResponse($data,200);
    }

    public function createTask(Request $request) {
        try {
            DB::beginTransaction();
            $this->validate($request, [
                'company_id' => 'required|string',
                'employee_id' => 'required|string',
                'project_id' => 'required|string',
                'title' => 'required|string',
                'html_description' => 'required|string',
                'project_section_id' => 'string',
                'assignee_id' => 'string',
                'due_at' => 'date_format:d-m-Y',
                'completed_at' => 'date_format:d-m-Y',
            ]);

            if(isset($request->completed) && $request->completed==1) {
                $completed = 1;
                $completed_at = date('Y-m-d H:i:s');
                //$completed_at = $request->completed?Carbon::createFromFormat('d-m-Y', $request->completed_at)->format('Y-m-d'):date('Y-m-d');
                $completed_by_id = $request->employee_id;
            } else {
                $completed = 0;
                $completed_at = null;
                $completed_by_id = '';
            }

            $task_id = Str::uuid();
            $data = [
                'company_id' => $request->company_id,
                'project_id' => $request->project_id,
                'task_id' => $task_id,
                'title' => $request->title,
                'html_description' => $request->html_description,
                'description' => strip_tags($request->html_description),
                'assignee_id' => ($request->assignee_id?$request->assignee_id:$request->employee_id),
                'due_at' => ($request->due_at?Carbon::createFromFormat('d-m-Y', $request->transaction_date)->format('Y-m-d'):null),
                'tags' => ($request->tags?$request->tags:''),
                'project_section_id' => ($request->project_section_id?$request->project_section_id:''),
                'created_at' => date('Y-m-d H:i:s'),
                'created_by_id' => $request->employee_id,
                'completed' => $completed,
                'completed_at' => $completed_at,
                'completed_by_id' => $completed_by_id,
            ];

            Task::insert($data);
            DB::commit();

            $task = Task::find($task_id);
            $project = Project::find($task->project_id);
            $project_sections = ProjectSection::where('project_id','=',$task->project_id)->orderBy('sort_order')->get();
            $team_members = TeamMember::where('team_id','=',$project->team_id)->get();
            $arrMembers = [];
            foreach($team_members as $member) {
                $employee = Employee::find($member->employee_id);
                $arrMembers[] = [
                    'employee_id' => $employee->employee_id,
                    'employee_code' => $employee->employee_code,
                    'full_name' => $employee->full_name,
                    'email' => $employee->email,
                    'mobile_no' => $employee->mobile_no,
                ];
            }

            $arrSections = [];
            foreach($project_sections as $section) {
                $arrSections[] = [
                    'project_section_id' => $section->project_section_id,
                    'section_name' => $section->section_name,
                    'sort_order' => $section->sort_order,
                ];
            }

            if($task->project_section_id) {
                $objSection = ProjectSection::find($task->project_section_id);
                $arrCurrentSection = [
                    'project_section_id' => $objSection->project_section_id,
                    'section_name' => $objSection->section_name,
                ];
            } else {
                $arrCurrentSection = [];
            }

            if($task->assignee_id) {
                $objAssignee = Employee::find($task->assignee_id);
                $arrAssignee = [
                    'employee_id' => $objAssignee->employee_id,
                    'employee_code' => $objAssignee->employee_code,
                    'full_name' => $objAssignee->full_name,
                    'email' => $objAssignee->email,
                    'mobile_no' => $objAssignee->mobile_no,
                ];
            } else {
                $arrAssignee = [];
            }

            if($task->completed_by_id) {
                $objCompletedBy = Employee::find($task->completed_by_id);
                $arrCompletedBy = [
                    'employee_id' => $objCompletedBy->employee_id,
                    'employee_code' => $objCompletedBy->employee_code,
                    'full_name' => $objCompletedBy->full_name,
                    'email' => $objCompletedBy->email,
                    'mobile_no' => $objCompletedBy->mobile_no,
                ];
            } else {
                $arrCompletedBy = [];
            }

            if($task->created_by_id) {
                $objCreatedBy = Employee::find($task->created_by_id);
                $arrCreatedBy = [
                    'employee_id' => $objCreatedBy->employee_id,
                    'employee_code' => $objCreatedBy->employee_code,
                    'full_name' => $objCreatedBy->full_name,
                    'email' => $objCreatedBy->email,
                    'mobile_no' => $objCreatedBy->mobile_no,
                ];
            } else {
                $arrCreatedBy = [];
            }

            $arrComments = [];
            $objComments = TaskComment::where('task_id','=',$task_id)->orderBy('created_at')->get();
            foreach($objComments as $comment) {
                $arrComments[] = [
                    'task_comment_id' => $comment->task_comment_id,
                    'comment' => $comment->comment,
                    'system_generated' => $comment->system_generated,
                    'created_at' => Carbon::createFromFormat('Y-m-d H:i:s', $comment->created_at)->format('d-m-Y H:i:s'),
                ];
            }

            $data = [
                'task_id' => $task->task_id,
                'title' => $task->title,
                'html_description' => $task->html_description,
                'description' => $task->description,
                'current_section' => $arrCurrentSection,
                'assignee' => $arrAssignee,
                'due_at' => $task->due_at,
                'tags' => $task->tags,
                'sections' => $arrSections,
                'completed' => $task->completed,
                'completed_at' => $task->completed_at,
                'completed_by' => $arrCompletedBy,
                'created_by' => $arrCreatedBy,
                'comments' => $arrComments,
            ];

            return $this->jsonResponse($data,200);
        } catch(Exception $e) {
            DB::rollBack();
            return $this->jsonResponse([],$e->getCode(),$e->getMessage());
        }
    }

    public function getSectionsFromProject(Request $request, $project_id) {
        $this->validate($request, [
            'company_id' => 'required|string',
            'employee_id' => 'required|string',
        ]);

        $company_id = $request->company_id;
        $employee_id = $request->employee_id;

        $objValid = TeamMember::join('tm_team','tm_team.team_id','=','tm_team_member.team_id')
            ->join('tm_project','tm_project.team_id','=','tm_team.team_id')
            ->join('tm_task','tm_task.project_id','=','tm_project.project_id')
            ->where('tm_team_member.employee_id','=',$employee_id)
            ->where('tm_project.company_id','=',$company_id)
            ->first();

        if($objValid) {

            $arrSections = [];
            $arrSections['untitled'] = [
                'project_section_id' => 'untitled',
                'section_name' => 'untitled section',
                'sort_order' => 0,
            ];
            $sections = ProjectSection::where('project_id','=',$project_id)->orderBy('sort_order')->get();
            foreach($sections as $section) {

                $arrSections[$section['section_name']] = [
                    'project_section_id' => $section->project_section_id,
                    'section_name' => $section->section_name,
                    'sort_order' => $section->sort_order,
                ];
            }

            return $this->jsonResponse(array_values($arrSections),200);
        } else {
            return $this->jsonResponse([],403, 'Invalid Request');
        }
    }

    public function getTasksFromSection(Request $request, $project_section_id) {
        $this->validate($request, [
            'company_id' => 'required|string',
            'employee_id' => 'required|string',
            'project_id' => 'required|string',
            'completed' => 'in:0,1',
            'order_by' => 'string',
            'sort_by' => 'string',
            'page' => 'integer',
            'limit' => 'integer',
        ]);
        $project_section_id = ($project_section_id=='untitled'?'':$project_section_id);

        $company_id = $request->company_id;
        $employee_id = $request->employee_id;
        $completed = $request->completed;
        $project_id = $request->project_id;

        $objValid = TeamMember::join('tm_team','tm_team.team_id','=','tm_team_member.team_id')
            ->join('tm_project','tm_project.team_id','=','tm_team.team_id')
            ->join('tm_task','tm_task.project_id','=','tm_project.project_id')
            ->where('tm_team_member.employee_id','=',$employee_id)
            ->where('tm_project.company_id','=',$company_id)
            ->where('tm_project.project_id','=',$project_id)
            ->first();
        if($objValid) {
            $objTasks = TeamMember::join('tm_team','tm_team.team_id','=','tm_team_member.team_id')
                ->join('tm_project','tm_project.team_id','=','tm_team.team_id')
                ->join('tm_task','tm_task.project_id','=','tm_project.project_id')
                ->leftJoin('tm_project_section', 'tm_project_section.project_section_id', '=', 'tm_task.project_section_id')
                ->where('tm_team_member.employee_id','=',$employee_id)
                ->where('tm_project.company_id','=',$company_id)
                ->where('tm_project.project_id','=',$project_id)
                ->where('tm_task.project_section_id','=',$project_section_id)
                ->select('tm_project_section.project_section_id','tm_project_section.sort_order', DB::raw('IFNULL(tm_project_section.section_name,"untitled section") as section_name'),'tm_task.task_id', 'tm_task.title', 'tm_task.description', 'tm_task.assignee_id', 'tm_task.due_at', 'tm_task.tags','tm_task.completed', 'tm_task.completed_at', 'tm_task.completed_by_id','tm_task.created_at', 'tm_task.created_by_id')
                ->orderBy('tm_project_section.sort_order')
            ;
            if(isset($request->completed)) {
                $objTasks = $objTasks->where('tm_task.completed','=',$completed);
            }
            $tasks = $objTasks->get();

            foreach ($tasks as $task) {
                $task_id = $task->task_id;

                if($task->assignee_id) {
                    $objAssignee = Employee::find($task->assignee_id);
                    $arrAssignee = [
                        'employee_id' => $objAssignee->employee_id,
                        'employee_code' => $objAssignee->employee_code,
                        'full_name' => $objAssignee->full_name,
                        'email' => $objAssignee->email,
                        'mobile_no' => $objAssignee->mobile_no,
                    ];
                } else {
                    $arrAssignee = [];
                }

                if($task->completed_by_id) {
                    $objCompletedBy = Employee::find($task->completed_by_id);
                    $arrCompletedBy = [
                        'employee_id' => $objCompletedBy->employee_id,
                        'employee_code' => $objCompletedBy->employee_code,
                        'full_name' => $objCompletedBy->full_name,
                        'email' => $objCompletedBy->email,
                        'mobile_no' => $objCompletedBy->mobile_no,
                    ];
                } else {
                    $arrCompletedBy = [];
                }

                if($task->created_by_id) {
                    $objCreatedBy = Employee::find($task->created_by_id);
                    $arrCreatedBy = [
                        'employee_id' => $objCreatedBy->employee_id,
                        'employee_code' => $objCreatedBy->employee_code,
                        'full_name' => $objCreatedBy->full_name,
                        'email' => $objCreatedBy->email,
                        'mobile_no' => $objCreatedBy->mobile_no,
                    ];
                } else {
                    $arrCreatedBy = [];
                }

                $arrTasks[] = [
                    'task_id' => $task->task_id,
                    'title' => $task->title,
                    'html_description' => $task->html_description,
                    'description' => $task->description,
                    'assignee' => $arrAssignee,
                    'due_at' => $task->due_at,
                    'tags' => $task->tags,
                    'completed' => $task->completed,
                    'completed_at' => $task->completed_at,
                    'completed_by' => $arrCompletedBy,
                    'created_by' => $arrCreatedBy,
                    //'comments' => $arrComments,
                ];
            }

            if($project_section_id=='') {
                $data = [
                    'project_section_id' => 'untitled',
                    'section_name' => 'untitled section',
                    'sort_order' => 0,
                    'tasks' => $arrTasks
                ];
            } else {
                $section = ProjectSection::find($project_section_id);
                $data = [
                    'project_section_id' => $section->project_section_id,
                    'section_name' => $section->section_name,
                    'sort_order' => $section->sort_order,
                    'tasks' => $arrTasks
                ];
            }
            //return response()->json(['data' => $data], 200);
            return $this->jsonResponse($data,200);
        } else {
            return $this->jsonResponse([],403, 'Invalid Request');
        }
    }

    public function getTasksFromProject(Request $request, $project_id) {
        $this->validate($request, [
            'company_id' => 'required|string',
            'employee_id' => 'required|string',
            'completed' => 'in:0,1',
            'order_by' => 'string',
            'sort_by' => 'string',
            'page' => 'integer',
            'limit' => 'integer',
        ]);

        $company_id = $request->company_id;
        $employee_id = $request->employee_id;
        $completed = $request->completed;

        $objValid = TeamMember::join('tm_team','tm_team.team_id','=','tm_team_member.team_id')
            ->join('tm_project','tm_project.team_id','=','tm_team.team_id')
            ->join('tm_task','tm_task.project_id','=','tm_project.project_id')
            ->where('tm_team_member.employee_id','=',$employee_id)
            ->where('tm_project.company_id','=',$company_id)
            ->first();
        if($objValid) {
            $objTasks = TeamMember::join('tm_team','tm_team.team_id','=','tm_team_member.team_id')
                ->join('tm_project','tm_project.team_id','=','tm_team.team_id')
                ->join('tm_task','tm_task.project_id','=','tm_project.project_id')
                ->leftJoin('tm_project_section', 'tm_project_section.project_section_id', '=', 'tm_task.project_section_id')
                ->where('tm_team_member.employee_id','=',$employee_id)
                ->where('tm_project.company_id','=',$company_id)
                ->where('tm_project.project_id','=',$project_id)
                ->select('tm_project_section.project_section_id','tm_project_section.sort_order', DB::raw('IFNULL(tm_project_section.section_name,"untitled section") as section_name'),'tm_task.task_id', 'tm_task.title', 'tm_task.description', 'tm_task.assignee_id', 'tm_task.due_at', 'tm_task.tags','tm_task.completed', 'tm_task.completed_at', 'tm_task.completed_by_id','tm_task.created_at', 'tm_task.created_by_id')
                ->orderBy('tm_project_section.sort_order')
            ;
            if(isset($request->completed)) {
                $objTasks = $objTasks->where('tm_task.completed','=',$completed);
            }
            $tasks = $objTasks->get();

            $arrSections = [];
            $sections = ProjectSection::where('project_id','=',$project_id)->orderBy('sort_order')->get();
            foreach($sections as $section) {
                $arrSections[$section['section_name']] = [
                    'project_section_id' => $section->project_section_id,
                    'section_name' => $section->section_name,
                    'sort_order' => $section->sort_order,
                    'tasks' => []
                ];
            }
            foreach ($tasks as $task) {
                $task_id = $task->task_id;
                $section_name = $task->section_name;
                if(!isset($arrSections[$section_name])) {
                    $new_array[$section_name] = [
                        'project_section_id' => 'untitled',
                        'section_name' => $section_name,
                        'sort_order' => 0,
                        'tasks' => []
                    ];
                    $arrSections = array_merge($new_array,$arrSections);
                }

//                $arrComments = [];
//                $objComments = TaskComment::where('task_id','=',$task_id)->orderBy('created_at')->get();
//                foreach($objComments as $comment) {
//                    $arrComments[] = [
//                        'task_comment_id' => $comment->task_comment_id,
//                        'comment' => $comment->comment,
//                        'system_generated' => $comment->system_generated,
//                        'created_at' => Carbon::createFromFormat('Y-m-d H:i:s', $comment->created_at)->format('d-m-Y H:i:s'),
//                    ];
//                }

                if($task->assignee_id) {
                    $objAssignee = Employee::find($task->assignee_id);
                    $arrAssignee = [
                        'employee_id' => $objAssignee->employee_id,
                        'employee_code' => $objAssignee->employee_code,
                        'full_name' => $objAssignee->full_name,
                        'email' => $objAssignee->email,
                        'mobile_no' => $objAssignee->mobile_no,
                    ];
                } else {
                    $arrAssignee = [];
                }

                if($task->completed_by_id) {
                    $objCompletedBy = Employee::find($task->completed_by_id);
                    $arrCompletedBy = [
                        'employee_id' => $objCompletedBy->employee_id,
                        'employee_code' => $objCompletedBy->employee_code,
                        'full_name' => $objCompletedBy->full_name,
                        'email' => $objCompletedBy->email,
                        'mobile_no' => $objCompletedBy->mobile_no,
                    ];
                } else {
                    $arrCompletedBy = [];
                }

                if($task->created_by_id) {
                    $objCreatedBy = Employee::find($task->created_by_id);
                    $arrCreatedBy = [
                        'employee_id' => $objCreatedBy->employee_id,
                        'employee_code' => $objCreatedBy->employee_code,
                        'full_name' => $objCreatedBy->full_name,
                        'email' => $objCreatedBy->email,
                        'mobile_no' => $objCreatedBy->mobile_no,
                    ];
                } else {
                    $arrCreatedBy = [];
                }

                $arrSections[$section_name]['tasks'][] = [
                    'task_id' => $task->task_id,
                    'title' => $task->title,
                    'html_description' => $task->html_description,
                    'description' => $task->description,
                    'assignee' => $arrAssignee,
                    'due_at' => $task->due_at,
                    'tags' => $task->tags,
                    'completed' => $task->completed,
                    'completed_at' => $task->completed_at,
                    'completed_by' => $arrCompletedBy,
                    'created_by' => $arrCreatedBy,
                    //'comments' => $arrComments,
                ];
            }

            return $this->jsonResponse(array_values($arrSections),200);
        } else {
            return $this->jsonResponse([],403,'Invalid Request');
        }
    }

    public function getTask(Request $request) {
        $this->validate($request, [
            'company_id' => 'required|string',
            'employee_id' => 'required|string',
            'task_id' => 'required|string',
        ]);

        $company_id = $request->company_id;
        $employee_id = $request->employee_id;
        $task_id = $request->task_id;

        $objValid = TeamMember::join('tm_team','tm_team.team_id','=','tm_team_member.team_id')
            ->join('tm_project','tm_project.team_id','=','tm_team.team_id')
            ->join('tm_task','tm_task.project_id','=','tm_project.project_id')
            ->where('tm_team_member.employee_id','=',$employee_id)
            ->where('tm_project.company_id','=',$company_id)
            ->where('tm_task.task_id','=',$task_id)
            ->first();
        if($objValid) {
            $task = Task::find($task_id);
            $project = Project::find($task->project_id);
            $project_sections = ProjectSection::where('project_id','=',$task->project_id)->orderBy('sort_order')->get();
            $team_members = TeamMember::where('team_id','=',$project->team_id)->get();
            $arrMembers = [];
            foreach($team_members as $member) {
                $employee = Employee::find($member->employee_id);
                $arrMembers[] = [
                    'employee_id' => $employee->employee_id,
                    'employee_code' => $employee->employee_code,
                    'full_name' => $employee->full_name,
                    'email' => $employee->email,
                    'mobile_no' => $employee->mobile_no,
                ];
            }

            $arrComments = [];
            $objComments = TaskComment::where('task_id','=',$task_id)->orderBy('created_at')->get();
            foreach($objComments as $comment) {
                $employee = Employee::find($comment->created_by_id);
                if($employee) {
                    $created_by = [
                        'employee_id' => $employee->employee_id,
                        'employee_code' => $employee->employee_code,
                        'full_name' => $employee->full_name,
                        'email' => $employee->email,
                        'mobile_no' => $employee->mobile_no,
                    ];
                } else {
                    $created_by = [];
                }
                $arrComments[] = [
                    'task_comment_id' => $comment->task_comment_id,
                    'comment' => $comment->comment,
                    'system_generated' => $comment->system_generated,
                    'created_at' => Carbon::createFromFormat('Y-m-d H:i:s', $comment->created_at)->format('d-m-Y H:i:s'),
                    'created_by' => $created_by
                ];
            }

            $arrSections = [];
            foreach($project_sections as $section) {
                $arrSections[] = [
                    'project_section_id' => $section->project_section_id,
                    'section_name' => $section->section_name,
                    'sort_order' => $section->sort_order,
                ];
            }

            if($task->project_section_id) {
                $objSection = ProjectSection::find($task->project_section_id);
                $arrCurrentSection = [
                    'project_section_id' => $objSection->project_section_id,
                    'section_name' => $objSection->section_name,
                ];
            } else {
                $arrCurrentSection = [];
            }

            if($task->assignee_id) {
                $objAssignee = Employee::find($task->assignee_id);
                $arrAssignee = [
                    'employee_id' => $objAssignee->employee_id,
                    'employee_code' => $objAssignee->employee_code,
                    'full_name' => $objAssignee->full_name,
                    'email' => $objAssignee->email,
                    'mobile_no' => $objAssignee->mobile_no,
                ];
            } else {
                $arrAssignee = [];
            }

            if($task->completed_by_id) {
                $objCompletedBy = Employee::find($task->completed_by_id);
                $arrCompletedBy = [
                    'employee_id' => $objCompletedBy->employee_id,
                    'employee_code' => $objCompletedBy->employee_code,
                    'full_name' => $objCompletedBy->full_name,
                    'email' => $objCompletedBy->email,
                    'mobile_no' => $objCompletedBy->mobile_no,
                ];
            } else {
                $arrCompletedBy = [];
            }

            if($task->created_by_id) {
                $objCreatedBy = Employee::find($task->created_by_id);
                $arrCreatedBy = [
                    'employee_id' => $objCreatedBy->employee_id,
                    'employee_code' => $objCreatedBy->employee_code,
                    'full_name' => $objCreatedBy->full_name,
                    'email' => $objCreatedBy->email,
                    'mobile_no' => $objCreatedBy->mobile_no,
                ];
            } else {
                $arrCreatedBy = [];
            }

            $data = [
                'task_id' => $task->task_id,
                'title' => $task->title,
                'html_description' => $task->html_description,
                'description' => $task->description,
                'current_section' => $arrCurrentSection,
                'assignee' => $arrAssignee,
                'due_at' => $task->due_at,
                'tags' => $task->tags,
                'completed' => $task->completed,
                'completed_at' => $task->completed_at,
                'completed_by' => $arrCompletedBy,
                'created_by' => $arrCreatedBy,
                'comments' => $arrComments,
                'sections' => $arrSections,
                'members' => $arrMembers,
            ];
            return $this->jsonResponse($data,200);
        } else {
            return $this->jsonResponse([],403,'Invalid Request');
        }
    }

    public function updateTask(Request $request, $id) {
        try {
            $this->validate($request, [
                'company_id' => 'required|string',
                'employee_id' => 'required|string',
                'project_id' => 'required|string',
                'title' => 'string',
                'html_description' => 'string',
                'assignee_id' => 'string',
                'project_section_id' => 'string',
                'due_at' => 'date_format:d-m-Y',
                'completed' => 'in:0,1',
            ]);

            $objTask = Task::find($id);
            if($objTask) {
                $data = [];
                $arrTaskComments = [];

                if(isset($request->title) && $request->title != '') {
                    $objTask->title = $request->title;
                }

                if(isset($request->html_description) && $request->html_description != '') {
                    $objTask->html_description = $request->html_description;
                    $objTask->description = strip_tags($request->html_description);
                }

                if(isset($request->assignee_id) && $request->assignee_id != '') {
                    $objTask->assignee_id = $request->assignee_id;

                    $objEmployee = Employee::find($request->employee_id);
                    $objAssignee = Employee::find($request->assignee_id);

                    $arrTaskComments[] = [
                        'company_id' => $request->company_id,
                        'project_id' => $request->project_id,
                        'task_id' => $id,
                        'task_comment_id' => Str::uuid(),
                        'system_generated' => 1,
                        'comment' => '`'.$objEmployee->full_name.'` assigned to `'.$objAssignee->full_name.'`.',
                        'created_at' => date('Y-m-d H:i:s'),
                        'created_by_id' => $request->employee_id,
                    ];
                } elseif (isset($request->assignee_id) && $request->assignee_id == '') {
                    $objTask->assignee_id = $request->assignee_id;

                    $objEmployee = Employee::find($request->employee_id);

                    $arrTaskComments[] = [
                        'company_id' => $request->company_id,
                        'project_id' => $request->project_id,
                        'task_id' => $id,
                        'task_comment_id' => Str::uuid(),
                        'system_generated' => 1,
                        'comment' => '`'.$objEmployee->full_name.'` unassigned this task.',
                        'created_at' => date('Y-m-d H:i:s'),
                        'created_by_id' => $request->employee_id,
                    ];
                }

                if(isset($request->project_section_id) && $request->project_section_id != '') {
                    $objTask->project_section_id = $request->project_section_id;

                    $objEmployee = Employee::find($request->employee_id);
                    $objSection = ProjectSection::find($request->project_section_id);

                    $arrTaskComments[] = [
                        'company_id' => $request->company_id,
                        'project_id' => $request->project_id,
                        'task_id' => $id,
                        'task_comment_id' => Str::uuid(),
                        'system_generated' => 1,
                        'comment' => '`'.$objEmployee->full_name.'` move this task to `'.$objSection->section_name.'`.',
                        'created_at' => date('Y-m-d H:i:s'),
                        'created_by_id' => $request->employee_id,
                    ];
                } elseif (isset($request->project_section_id) && $request->project_section_id == '') {
                    $objTask->project_section_id = $request->project_section_id;

                    $objEmployee = Employee::find($request->employee_id);

                    $arrTaskComments[] = [
                        'company_id' => $request->company_id,
                        'project_id' => $request->project_id,
                        'task_id' => $id,
                        'task_comment_id' => Str::uuid(),
                        'system_generated' => 1,
                        'comment' => '`'.$objEmployee->full_name.'` removed this task from section.',
                        'created_at' => date('Y-m-d H:i:s'),
                        'created_by_id' => $request->employee_id,
                    ];
                }

                if(isset($request->due_at) && $request->due_at != '') {
                    $objTask->due_at = Carbon::createFromFormat('d-m-Y', $request->due_at)->format('Y-m-d');

                    $objEmployee = Employee::find($request->employee_id);

                    $arrTaskComments[] = [
                        'company_id' => $request->company_id,
                        'project_id' => $request->project_id,
                        'task_id' => $id,
                        'task_comment_id' => Str::uuid(),
                        'system_generated' => 1,
                        'comment' => '`'.$objEmployee->full_name.'` changed Due Date to `'.$request->due_at.'`.',
                        'created_at' => date('Y-m-d H:i:s'),
                        'created_by_id' => $request->employee_id,
                    ];
                } elseif (isset($request->due_at) && $request->due_at == '') {
                    $objTask->due_at = NULL;

                    $objEmployee = Employee::find($request->employee_id);

                    $arrTaskComments[] = [
                        'company_id' => $request->company_id,
                        'project_id' => $request->project_id,
                        'task_id' => $id,
                        'task_comment_id' => Str::uuid(),
                        'system_generated' => 1,
                        'comment' => '`'.$objEmployee->full_name.'` removed due date.',
                        'created_at' => date('Y-m-d H:i:s'),
                        'created_by_id' => $request->employee_id,
                    ];
                }

                if(isset($request->completed) && $request->completed==1) {
                    $objTask->completed = 1;
                    $objTask->completed_at = date('Y-m-d H:i:s');
                    $objTask->completed_by_id = $request->employee_id;

                    $objEmployee = Employee::find($request->employee_id);

                    $arrTaskComments[] = [
                        'company_id' => $request->company_id,
                        'project_id' => $request->project_id,
                        'task_id' => $id,
                        'task_comment_id' => Str::uuid(),
                        'system_generated' => 1,
                        'comment' => '`'.$objEmployee->full_name.'` completed this task.',
                        'created_at' => date('Y-m-d H:i:s'),
                        'created_by_id' => $request->employee_id,
                    ];
                } elseif(isset($request->completed) && $request->completed==0) {
                    $objTask->completed = 0;
                    $objTask->completed_at = null;
                    $objTask->completed_by_id = '';

                    $objEmployee = Employee::find($request->employee_id);

                    $arrTaskComments[] = [
                        'company_id' => $request->company_id,
                        'project_id' => $request->project_id,
                        'task_id' => $id,
                        'task_comment_id' => Str::uuid(),
                        'system_generated' => 1,
                        'comment' => '`'.$objEmployee->full_name.'` reopened this task.',
                        'created_at' => date('Y-m-d H:i:s'),
                        'created_by_id' => $request->employee_id,
                    ];
                }

//                $objTask->save();
//                $arrChanges = $objTask->getChanges();
//                return response()->json(['data' => $arrChanges], 200);
                DB::beginTransaction();
                $objTask->save();
                foreach($arrTaskComments as $comment) {
                    TaskComment::insert($comment);
                }
                DB::commit();

                $project = Project::find($objTask->project_id);
                $project_sections = ProjectSection::where('project_id','=',$objTask->project_id)->orderBy('sort_order')->get();
                $team_members = TeamMember::where('team_id','=',$project->team_id)->get();
                $arrMembers = [];
                foreach($team_members as $member) {
                    $employee = Employee::find($member->employee_id);
                    $arrMembers[] = [
                        'employee_id' => $employee->employee_id,
                        'employee_code' => $employee->employee_code,
                        'full_name' => $employee->full_name,
                        'email' => $employee->email,
                        'mobile_no' => $employee->mobile_no,
                    ];
                }

                $arrSections = [];
                foreach($project_sections as $section) {
                    $arrSections[] = [
                        'project_section_id' => $section->project_section_id,
                        'section_name' => $section->section_name,
                        'sort_order' => $section->sort_order,
                    ];
                }

//                $arrComments = [];
//                $objComments = TaskComment::where('task_id','=',$id)->orderBy('created_at')->get();
//                foreach($objComments as $comment) {
//                    $arrComments[] = [
//                        'task_comment_id' => $comment->task_comment_id,
//                        'comment' => $comment->comment,
//                        'system_generated' => $comment->system_generated,
//                        'created_at' => Carbon::createFromFormat('Y-m-d H:i:s', $comment->created_at)->format('d-m-Y H:i:s'),
//                    ];
//                }
                $objComments = TaskComment::where('task_id','=',$id)->orderBy('created_at')->get();
                $arrComments = [];
                foreach($objComments as $comment) {
                    $employee = Employee::find($comment->created_by_id);
                    if($employee) {
                        $created_by = [
                            'employee_id' => $employee->employee_id,
                            'employee_code' => $employee->employee_code,
                            'full_name' => $employee->full_name,
                            'email' => $employee->email,
                            'mobile_no' => $employee->mobile_no,
                        ];
                    } else {
                        $created_by = [];
                    }
                    $arrComments[] = [
                        'task_comment_id' => $comment->task_comment_id,
                        'comment' => $comment->comment,
                        'system_generated' => $comment->system_generated,
                        'created_at' => Carbon::createFromFormat('Y-m-d H:i:s', $comment->created_at)->format('d-m-Y H:i:s'),
                        'created_by' => $created_by
                    ];
                }

                if($objTask->project_section_id) {
                    $objSection = ProjectSection::find($objTask->project_section_id);
                    $arrCurrentSection = [
                        'project_section_id' => $objSection->project_section_id,
                        'section_name' => $objSection->section_name,
                    ];
                } else {
                    $arrCurrentSection = [];
                }

                if($objTask->assignee_id) {
                    $objAssignee = Employee::find($objTask->assignee_id);
                    $arrAssignee = [
                        'employee_id' => $objAssignee->employee_id,
                        'employee_code' => $objAssignee->employee_code,
                        'full_name' => $objAssignee->full_name,
                        'email' => $objAssignee->email,
                        'mobile_no' => $objAssignee->mobile_no,
                    ];
                } else {
                    $arrAssignee = [];
                }

                if($objTask->completed_by_id) {
                    $objCompletedBy = Employee::find($objTask->completed_by_id);
                    $arrCompletedBy = [
                        'employee_id' => $objCompletedBy->employee_id,
                        'employee_code' => $objCompletedBy->employee_code,
                        'full_name' => $objCompletedBy->full_name,
                        'email' => $objCompletedBy->email,
                        'mobile_no' => $objCompletedBy->mobile_no,
                    ];
                } else {
                    $arrCompletedBy = [];
                }

                if($objTask->created_by_id) {
                    $objCreatedBy = Employee::find($objTask->created_by_id);
                    $arrCreatedBy = [
                        'employee_id' => $objCreatedBy->employee_id,
                        'employee_code' => $objCreatedBy->employee_code,
                        'full_name' => $objCreatedBy->full_name,
                        'email' => $objCreatedBy->email,
                        'mobile_no' => $objCreatedBy->mobile_no,
                    ];
                } else {
                    $arrCreatedBy = [];
                }

                $data = [
                    'task_id' => $objTask->task_id,
                    'title' => $objTask->title,
                    'html_description' => $objTask->html_description,
                    'description' => $objTask->description,
                    'current_section' => $arrCurrentSection,
                    'assignee' => $arrAssignee,
                    'due_at' => $objTask->due_at,
                    'tags' => $objTask->tags,
                    'sections' => $arrSections,
                    'completed' => $objTask->completed,
                    'completed_at' => $objTask->completed_at,
                    'completed_by' => $arrCompletedBy,
                    'created_by' => $arrCreatedBy,
                    'comments' => $arrComments,
                ];
                return $this->jsonResponse($data,200);
            } else {
                return $this->jsonResponse([],403,'Invalid Request');
            }
        } catch(Exception $e) {
            DB::rollBack();
            //return response()->json($e->getMessage(), $e->getCode());
            return $this->jsonResponse([],$e->getCode(),$e->getMessage());
        }
    }

    public function addComment(Request $request) {
        try {
            $this->validate($request, [
                'company_id' => 'required|string',
                'employee_id' => 'required|string',
                'project_id' => 'required|string',
                'task_id' => 'required|string',
                'comment' => 'required|string',
                'hour' => 'integer',
                'minute' => 'integer',
            ]);

            $objTask = Task::where('company_id','=',$request->company_id)
                ->where('project_id','=',$request->project_id)
                ->where('task_id','=',$request->task_id)
                ->first();
            if($objTask) {
                $data = [
                    'company_id' => $request->company_id,
                    'project_id' => $request->project_id,
                    'task_id' => $request->task_id,
                    'task_comment_id' => Str::uuid(),
                    'comment' => $request->comment,
                    'system_generated' => 0,
                    'created_at' => date('Y-m-d H:i:s'),
                    'created_by_id' => $request->employee_id,
                ];
                if(isset($request->hour)) {
                    $data['hour'] = $request->hour;
                } else {
                    $data['hour'] = 0;
                }
                if(isset($request->minute)) {
                    $data['minute'] = $request->minute;
                } else {
                    $data['minute'] = 0;
                }

                DB::beginTransaction();
                TaskComment::insert($data);
                DB::commit();

                $objComments = TaskComment::where('task_id','=',$request->task_id)->orderBy('created_at')->get();
                $arrComments = [];
                foreach($objComments as $comment) {
                    $employee = Employee::find($comment->created_by_id);
                    if($employee) {
                        $created_by = [
                            'employee_id' => $employee->employee_id,
                            'employee_code' => $employee->employee_code,
                            'full_name' => $employee->full_name,
                            'email' => $employee->email,
                            'mobile_no' => $employee->mobile_no,
                        ];
                    } else {
                        $created_by = [];
                    }
                    $arrComments[] = [
                        'task_comment_id' => $comment->task_comment_id,
                        'comment' => $comment->comment,
                        'system_generated' => $comment->system_generated,
                        'created_at' => Carbon::createFromFormat('Y-m-d H:i:s', $comment->created_at)->format('d-m-Y H:i:s'),
                        'created_by' => $created_by
                    ];
                }
                return $this->jsonResponse($arrComments,200);
            } else {
                return $this->jsonResponse([],403,'Invalid Request');
            }
        } catch (Exception $e) {
            DB::rollBack();
            return $this->jsonResponse([],$e->getCode(),$e->getMessage());
        }

    }
}