<?php

namespace App\Models;

use Mockery\Exception;
use Tymon\JWTAuth\JWT;
use Illuminate\Support\Str;
use Illuminate\Auth\Authenticatable;
use Laravel\Lumen\Auth\Authorizable;
use Illuminate\Database\Eloquent\Model;
use Tymon\JWTAuth\Contracts\JWTSubject;
use Illuminate\Contracts\Auth\Authenticatable as AuthenticatableContract;
use Illuminate\Contracts\Auth\Access\Authorizable as AuthorizableContract;


class Employee extends Model implements AuthenticatableContract, AuthorizableContract, JWTSubject
{
    use Authenticatable, Authorizable;

    protected $table = 'employee';
    protected $primaryKey = 'employee_id';
    public $timestamps = false;
    public $incrementing = false;
    protected $keyType = 'string';

    protected $hidden = [
        'login_password',
    ];

    /**
     * Get the identifier that will be stored in the subject claim of the JWT.
     *
     * @return mixed
     */
    public function getJWTIdentifier()
    {
        return $this->getKey();
    }

    /**
     * Return a key value array, containing any custom claims to be added to the JWT.
     *
     * @return array
     */
    public function getJWTCustomClaims()
    {
        return [];
    }

    public static function attempt($credentials)
    {
        try {
            $login_type = $credentials['login_type'];
            $username = $credentials['username'];
            $password = $credentials['password'];
            $employee = Employee::where($login_type,"=", $username)
                ->where('login_password',"=", $password)
                ->first();

            if($employee){

                // default 1 day expiration time for token
                $header = [
                    'alg' => 'HS256',
                    'typ' => 'JWT'
                ];

                $payload = [
                    'employee_id' => $employee->employee_id,
                    'full_name' => $employee->full_name,
                    'email' => $employee->email,
                    'exp' => time() * 60 * 24 // 1 day
                ];

                $token = self::generate_jwt($header, $payload);
                $employee->access_token = $token;
                $employee->save();
                return $employee;

            }
            return false;
        } catch (Exception $e) {
            die($e->getMessage());
        }
    }

    public static function generate_jwt($headers, $payload, $secret = 'secret') {

        $headers_encoded = self::base64url_encode(json_encode($headers));
        $payload_encoded = self::base64url_encode(json_encode($payload));
        $signature = hash_hmac('SHA256', "$headers_encoded.$payload_encoded", $secret, true);
        $signature_encoded = self::base64url_encode($signature);
        $jwt = "$headers_encoded.$payload_encoded.$signature_encoded";

        return $jwt;
    }

    public static function base64url_encode($str) {
        return rtrim(strtr(base64_encode($str), '+/', '-_'), '=');
    }


    public static function is_jwt_valid($jwt, $secret = 'secret') {
        // split the jwt
        $tokenParts = explode('.', $jwt);
        $header = base64_decode($tokenParts[0]);
        $payload = base64_decode($tokenParts[1]);
        $signature_provided = $tokenParts[2];


        // check the expiration time - note this will cause an error if there is no 'exp' claim in the jwt
        $expiration = json_decode($payload)->exp;
        $is_token_expired = ($expiration - time()) < 0;

        // build a signature based on the header and payload using the secret
        $base64_url_header = self::base64url_encode($header);
        $base64_url_payload = self::base64url_encode($payload);
        $signature = hash_hmac('SHA256', $base64_url_header . "." . $base64_url_payload, $secret, true);
        $base64_url_signature = self::base64url_encode($signature);

        // verify it matches the signature provided in the jwt
        $is_signature_valid = ($base64_url_signature === $signature_provided);

        if ($is_token_expired || !$is_signature_valid) {
            return FALSE;
        } else {
            return TRUE;
        }
    }

    public static function get_employee_id_from_jwt($token) {
        $tokenParts = explode('.', $token);
        $payload = base64_decode($tokenParts[1]);
        $payload = json_decode($payload);
        return $payload->employee_id;
    }

    public static function get_employee_from_jwt($token) {
        $employee_id = self::get_employee_id_from_jwt($token);
        return Employee::where('employee_', $employee_id)->first();
    }

    public static function refreshToken($employee_id) {

        $employee = Employee::where('employee_id', $employee_id)
            ->where('access_token', '<>', null)
            ->first();

        if($employee){

            // default 1 day expiration time for token
            $header = [
                'alg' => 'HS256',
                'typ' => 'JWT'
            ];

            $payload = [
                'employee_id' => $employee->employee_id,
                'full_name' => $employee->full_name,
                'email' => $employee->email,
                'exp' => time() * 60 * 24 // 1 day
            ];

            $token = self::generate_jwt($header, $payload);
            $employee->access_token = $token;
            $employee->save();
            return $employee;
        }

        return false;

    }
}